package viewcounts

import (
	"context"
	"time"

	"strconv"

	"fmt"

	"code.justin.tv/video/lvsapi/internal/caching/gocache"
	"github.com/cactus/go-statsd-client/statsd"
	cache "github.com/patrickmn/go-cache"
)

//InMemoryViewcountsCache wraps the cache implementation and usher backend
type InMemoryViewcountsCache struct {
	viewcountsBackend ViewcountsBackend
	cacheImpl         gocache.GoCache
	defaultExpiration time.Duration
	statter           statsd.Statter
}

//NewInMemoryViewcountsCache returns the wrapper which caches the viewcounts api
func NewInMemoryViewcountsCache(expiration, purgeInterval *time.Duration, viewCountsApiBaseUrl string, s statsd.Statter) ViewcountsCache {
	return &InMemoryViewcountsCache{
		viewcountsBackend: NewViewcountsClient(viewCountsApiBaseUrl, s),
		cacheImpl:         cache.New(*expiration, *purgeInterval),
		defaultExpiration: *expiration,
		statter:           s,
	}
}

//GetCount returns the channel count given a streamID
func (vcc *InMemoryViewcountsCache) GetCount(ctx context.Context, streamId uint64) (uint64, error) {
	vcc.incrementStat("viewcount_cache_get")
	count, err := vcc.get(ctx, streamId)
	if err != nil {
		return 0, err
	}

	return count, nil
}

//Check cache if it exists, if not query viewcount API and then update cache
func (vcc *InMemoryViewcountsCache) get(ctx context.Context, streamId uint64) (uint64, error) {
	cacheKey := getCacheKey(streamId)

	value, found := vcc.cacheImpl.Get(cacheKey)
	if found {
		vcc.incrementStat("viewcount_cache_hit")
		if count, ok := value.(uint64); ok {
			return count, nil
		}
		return 0, fmt.Errorf("returned cache value is not of expected type")
	}

	vcc.incrementStat("viewcount_cache_miss")

	//Get value from viewcount api and then update the cache
	count, err := vcc.viewcountsBackend.GetCountsForStream(ctx, streamId)
	if err != nil {
		return 0, fmt.Errorf("Viewcount API returned error :%s", err)
	}

	//Valid value , update cache and then return
	vcc.cacheImpl.Set(cacheKey, count.GetCount(), vcc.defaultExpiration)

	return count.GetCount(), nil

}

//The cache library being used only supports strings as keys, so convert the streamId to
func getCacheKey(streamId uint64) string {
	return strconv.FormatUint(streamId, 10)
}

func (vcc *InMemoryViewcountsCache) incrementStat(dimension string) {
	if vcc.statter != nil {
		_ = vcc.statter.Inc(dimension, 1, 1.0)
	}
}
