package viewcounts

import (
	"context"
	"fmt"
	"strings"
	"testing"
	"time"

	"github.com/stretchr/testify/require"

	"code.justin.tv/video/lvsapi/internal/caching/gocache/gocachefakes"
	"code.justin.tv/video/viewcount-api/api/viewcounts"
)

//We just make sure we get the correct value back when the entry is not present in cache
func TestSuccessfulGetViewcountAPI(t *testing.T) {
	vcBackend := &FakeViewcountsBackend{}
	vcBackend.GetCountsForStreamReturns(&viewcounts.Views{
		Count:           1,
		CountUnfiltered: 5,
	}, nil)

	cache := &gocachefakes.FakeGoCache{}
	cache.GetReturns(0, false)

	viewcountsCache := InMemoryViewcountsCache{
		viewcountsBackend: vcBackend,
		cacheImpl:         cache,
		defaultExpiration: time.Duration(2 * time.Second),
	}

	//Get should return the correct count value
	val, err := viewcountsCache.GetCount(context.Background(), 5)
	require.Nil(t, err)
	require.Equal(t, uint64(1), val)
	require.Equal(t, 1, vcBackend.GetCountsForStreamCallCount())

}

func TestViewcountAPIFailure(t *testing.T) {
	vcBackend := &FakeViewcountsBackend{}
	vcBackend.GetCountsForStreamReturns(nil, fmt.Errorf("404 stream id is invalid"))

	cache := &gocachefakes.FakeGoCache{}
	cache.GetReturns(0, false)

	viewcountsCache := InMemoryViewcountsCache{
		viewcountsBackend: vcBackend,
		cacheImpl:         cache,
		defaultExpiration: time.Duration(2 * time.Second),
	}

	//Get should return the correct count value
	val, err := viewcountsCache.GetCount(context.Background(), 5)
	require.Equal(t, uint64(0), val)
	require.NotNil(t, err)
	require.True(t, strings.Contains(err.Error(), "404 stream id is invalid"))
	require.Equal(t, 1, vcBackend.GetCountsForStreamCallCount())

}

func TestViewcountCacheSuccess(t *testing.T) {
	vcBackend := &FakeViewcountsBackend{}
	vcBackend.GetCountsForStreamReturns(nil, fmt.Errorf("404 stream id is invalid"))

	cache := &gocachefakes.FakeGoCache{}
	cache.GetReturns(uint64(5), true)

	viewcountsCache := InMemoryViewcountsCache{
		viewcountsBackend: vcBackend,
		cacheImpl:         cache,
		defaultExpiration: time.Duration(2 * time.Second),
	}

	//Get should return the correct count value
	val, err := viewcountsCache.GetCount(context.Background(), 5)
	require.Equal(t, uint64(5), val)
	require.Nil(t, err)
	require.Equal(t, 0, vcBackend.GetCountsForStreamCallCount())
}

func TestViewcountCacheSuccessInvalidFormat(t *testing.T) {
	vcBackend := &FakeViewcountsBackend{}
	vcBackend.GetCountsForStreamReturns(nil, fmt.Errorf("404 stream id is invalid"))

	cache := &gocachefakes.FakeGoCache{}
	cache.GetReturns(5, true)

	viewcountsCache := InMemoryViewcountsCache{
		viewcountsBackend: vcBackend,
		cacheImpl:         cache,
		defaultExpiration: time.Duration(2 * time.Second),
	}

	//Get should return the correct count value
	_, err := viewcountsCache.GetCount(context.Background(), 5)
	require.NotNil(t, err)
	require.True(t, strings.Contains(err.Error(), "returned cache value is not of expected type"))
	require.Equal(t, 0, vcBackend.GetCountsForStreamCallCount())

}
