package viewcounts

import (
	"context"
	"fmt"
	"strings"
	"testing"

	"github.com/stretchr/testify/require"

	"code.justin.tv/video/viewcount-api/api/viewcounts"
	"code.justin.tv/video/viewcount-api/api/viewcounts/viewcountsfakes"
)

func TestInvalidClient(t *testing.T) {
	client := &ViewcountsClient{}
	_, err := client.GetCountsForStream(context.Background(), 12334)
	require.NotNil(t, err)
	require.True(t, strings.Contains(err.Error(), "Invalid client"))
}

func TestNilStreamId(t *testing.T) {

	vcClient := &viewcountsfakes.FakeViewcounts{}

	client := ViewcountsClient{
		client: vcClient,
	}
	_, err := client.GetCountsForStream(context.Background(), 0)
	require.NotNil(t, err)
	require.True(t, strings.Contains(err.Error(), "Invalid Stream ID"))
}

func TestViewCountAPIError(t *testing.T) {
	fakeClient := &viewcountsfakes.FakeViewcounts{}
	fakeClient.ForStreamReturns(nil, fmt.Errorf("Internal error"))
	vcClient := &ViewcountsClient{
		client: fakeClient,
	}
	data, err := vcClient.GetCountsForStream(context.Background(), 12345)
	require.NotNil(t, err)
	require.True(t, strings.Contains(err.Error(), "Internal error"))
	require.Nil(t, data)
}

func TestValidViewCountResponse(t *testing.T) {
	fakeClient := &viewcountsfakes.FakeViewcounts{}
	fakeClient.ForStreamReturns(&viewcounts.Views{Count: 2, CountUnfiltered: 5}, nil)
	vcClient := &ViewcountsClient{
		client: fakeClient,
	}
	data, err := vcClient.GetCountsForStream(context.Background(), 12345)
	require.Nil(t, err)
	require.NotNil(t, data)
	require.Equal(t, uint64(2), data.GetCount())
	require.Equal(t, uint64(5), data.GetCountUnfiltered())

}
