package streamkey

import (
	"bytes"
	"context"
	"path"
	"sync"
	"time"

	"code.justin.tv/jorge/clock"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/s3"
	"github.com/aws/aws-sdk-go/service/s3/s3iface"
	"github.com/pkg/errors"
)

// S3SecretSourceConfig holds configuration values used when fetching secrets from S3
type S3SecretSourceConfig struct {
	// S3 should be an S3 client as constructed by s3.New()
	S3 s3iface.S3API

	// Clock is a clock interface used for testing primarily, should be left nil
	Clock clock.Clock

	// Bucket is the s3 bucket in which secret data will be stored and retrieved
	Bucket string
	// Prefix is the prefix for all secrets in s3
	Prefix string
	// CacheTimeout specifies the duration for which secrets should be cached
	CacheTimeout time.Duration
}

// S3SecretSource provides a KeyStore implementation which can
type S3SecretSource struct {
	cfg S3SecretSourceConfig

	fetchersMu sync.Mutex
	fetchers   map[string]*s3SecretFetcher
}

// NewS3SecretSource constructs a secret store which fetches and caches secrets from S3
func NewS3SecretSource(cfg S3SecretSourceConfig) *S3SecretSource {
	if cfg.Clock == nil {
		cfg.Clock = clock.New()
	}

	return &S3SecretSource{
		cfg:      cfg,
		fetchers: make(map[string]*s3SecretFetcher),
	}
}

func (ss *S3SecretSource) fetcher(customerID string) *s3SecretFetcher {
	ss.fetchersMu.Lock()
	defer ss.fetchersMu.Unlock()

	if ss.fetchers == nil {
		ss.fetchers = make(map[string]*s3SecretFetcher)
	}

	if _, ok := ss.fetchers[customerID]; !ok {
		ss.fetchers[customerID] = newS3SecretFetcher(customerID, ss.cfg)
	}

	return ss.fetchers[customerID]
}

// Get gets the secret for customerID and caches the value for up to CacheTimeout
// duration, returning stale values if the S3 fetch request fails.
func (ss *S3SecretSource) Get(ctx context.Context, customerID string) (*Secret, error) {
	return ss.fetcher(customerID).fetch(ctx)
}

// Set stores the secret for customerID and updates the cached value if the store
// operation succeeds.
func (ss *S3SecretSource) Set(ctx context.Context, customerID string, secret *Secret) error {
	return ss.fetcher(customerID).set(ctx, secret)
}

func newS3SecretFetcher(customerID string, cfg S3SecretSourceConfig) *s3SecretFetcher {
	return &s3SecretFetcher{customerID: customerID, cfg: cfg}
}

type s3SecretFetcher struct {
	customerID string
	cfg        S3SecretSourceConfig
	secret     *Secret
	fetchTime  time.Time
	sync.Mutex
}

func (f *s3SecretFetcher) fetchS3(ctx context.Context) (*Secret, error) {
	resp, err := f.cfg.S3.GetObjectWithContext(ctx, &s3.GetObjectInput{
		Bucket: aws.String(f.cfg.Bucket),
		Key:    aws.String(path.Join(f.cfg.Prefix, f.customerID+".pem")),
	})
	if err != nil {
		return nil, errors.Wrap(err, "s3 secret fetch failed")
	}
	defer func() {
		_ = resp.Body.Close()
	}()

	secret := new(Secret)
	if err := secret.Decode(resp.Body); err != nil {
		return nil, errors.Wrap(err, "s3 secret read failed")
	}

	return secret, nil
}

func (f *s3SecretFetcher) fetch(ctx context.Context) (*Secret, error) {
	f.Lock()
	defer f.Unlock()

	if f.secret == nil || f.cfg.Clock.Since(f.fetchTime) > f.cfg.CacheTimeout {
		secret, err := f.fetchS3(ctx)
		if err != nil {
			return nil, err
		}

		f.secret = secret
		f.fetchTime = f.cfg.Clock.Now()
	}

	return f.secret, nil
}

func (f *s3SecretFetcher) set(ctx context.Context, secret *Secret) error {
	f.Lock()
	defer f.Unlock()

	buf := &bytes.Buffer{}
	if err := secret.Encode(buf); err != nil {
		return errors.Wrap(err, "failed to pem encode secret for s3")
	}

	_, err := f.cfg.S3.PutObjectWithContext(ctx, &s3.PutObjectInput{
		Bucket: aws.String(f.cfg.Bucket),
		Key:    aws.String(path.Join(f.cfg.Prefix, f.customerID+".pem")),
		Body:   bytes.NewReader(buf.Bytes()),
	})
	if err != nil {
		return errors.Wrap(err, "s3 secret put failed")
	}

	f.secret = secret
	f.fetchTime = f.cfg.Clock.Now()

	return nil
}
