package streamkey

import (
	"crypto/rand"
	"encoding/pem"
	"fmt"
	"io"
	"io/ioutil"
	"os"

	"github.com/pkg/errors"
)

var (
	secretPemBlockType = "STREAMKEY SECRET"
)

// Secret is an opaque type which holds the key used to encrypt/decrypt
// stream key tokens
type Secret struct {
	bytes [32]byte
}

// Decode decodes a PEM encoded streamkey secret from a reader
func (k *Secret) Decode(r io.Reader) error {
	data, err := ioutil.ReadAll(r)
	if err != nil {
		return errors.Wrap(err, "failed to parse streamkey secret")
	}

	bl, _ := pem.Decode(data)
	if bl == nil {
		return errors.New("no PEM data found")
	}

	if bl.Type != secretPemBlockType {
		return fmt.Errorf("found unexpected PEM block type %s", bl.Type)
	}

	if len(bl.Bytes) != len(k.bytes) {
		return fmt.Errorf("found key of invalid size: %d", len(bl.Bytes))
	}

	copy(k.bytes[:], bl.Bytes)

	return nil
}

// Encode encodes and writes a PEM encoded streamkey secret a writer
func (k *Secret) Encode(w io.Writer) error {
	return pem.Encode(w, &pem.Block{
		Type:  secretPemBlockType,
		Bytes: k.bytes[:],
	})
}

// LoadSecret deserialize a PEM encoded streamkey secret from disk
func LoadSecret(filename string) (*Secret, error) {
	f, err := os.Open(filename)
	if err != nil {
		return nil, errors.Wrapf(err, "failed to open file %q for reading", filename)
	}
	defer func() { _ = f.Close() }()

	key := &Secret{}
	if err := key.Decode(f); err != nil {
		return nil, errors.Wrap(err, "failed to decode streamkey secret")
	}

	return key, nil
}

// SaveSecret serializes a PEM encoded streamkey secret to disk
func SaveSecret(filename string, k *Secret) error {
	f, err := os.Create(filename)
	if err != nil {
		return errors.Wrapf(err, "failed to open file %q for writing", filename)
	}
	defer func() { _ = f.Close() }()

	if err := k.Encode(f); err != nil {
		return errors.Wrap(err, "failed to encode streamkey secret")
	}

	return nil
}

// GenerateSecret generates a new streamkey secret using crypto/rand.Reader
func GenerateSecret() (*Secret, error) {
	var key Secret
	_, err := io.ReadFull(rand.Reader, key.bytes[:])
	if err != nil {
		return nil, errors.Wrap(err, "failed to read from random reader")
	}
	return &key, nil
}
