import cdk = require('@aws-cdk/core');
import ec2 = require('@aws-cdk/aws-ec2');
import iam = require('@aws-cdk/aws-iam');
import ddb = require('@aws-cdk/aws-dynamodb');
import lambda = require('@aws-cdk/aws-lambda');

export interface TurnipProps {
  invokeFunctionRoles: string[]
  canAccessSystemBindleLockID: string
  isAdminBindleLockID: string

}

export class Turnip extends cdk.Construct {
  ChannelsTable: ddb.Table
  ServiceLambda: lambda.Function

  constructor(scope: cdk.Construct, id: string, vpc: ec2.IVpc, props: TurnipProps) {
    super(scope, id);

    // Dynamo db table for tracking hidden channels
    this.ChannelsTable = new ddb.Table(this, 'ChannelsTable', {
      tableName: 'turnip-channels',
      partitionKey: { name: 'channel_id_hash', type: ddb.AttributeType.STRING },
      billingMode: ddb.BillingMode.PAY_PER_REQUEST,
    })

    this.ChannelsTable.addGlobalSecondaryIndex({
      indexName: 'idx_by_channel_id',
      partitionKey: { name: 'channel_id', type: ddb.AttributeType.STRING },
      projectionType: ddb.ProjectionType.ALL,
    })

    // Lambda for the service API
    this.ServiceLambda = new lambda.Function(this, 'Lambda', {
      vpc: vpc,
      runtime: lambda.Runtime.GO_1_X,
      memorySize: 1024,
      timeout: cdk.Duration.seconds(5),
      handler: 'bin/turnip-api',
      code: lambda.Code.asset('../bin/turnip-api.zip'),
      environment: {
        "channelsTableName": this.ChannelsTable.tableName,
        "canAccessSystemBindleLockID": props.canAccessSystemBindleLockID,
        "isAdminBindleLockID": props.isAdminBindleLockID,
      }
    })

    // Allow things to call the function
    for (let roleArn of props.invokeFunctionRoles) {
      this.ServiceLambda.grantInvoke(new iam.ArnPrincipal(roleArn))
    }

    this.ServiceLambda.addToRolePolicy(new iam.PolicyStatement({
      sid: 'DDBAccess',
      effect: iam.Effect.ALLOW,
      actions: [
        "dynamodb:PutItem",
        "dynamodb:DeleteItem",
        "dynamodb:UpdateItem",
        "dynamodb:GetItem",
        "dynamodb:Scan",
        "dynamodb:Query"
      ],
      resources: [
        this.ChannelsTable.tableArn,
        this.ChannelsTable.tableArn + '/*',
      ]
    }))

    this.ServiceLambda.addToRolePolicy(new iam.PolicyStatement({
      sid: 'BRASSAccess',
      effect: iam.Effect.ALLOW,
      actions: [
        "brassservice:IsAuthorized",
        "brassservice:BatchIsAuthorized",
      ],
      resources: [
        '*'
      ]
    }))
  }
}
