package main

import (
	"crypto/sha256"
	"encoding/csv"
	"fmt"
	"log"
	"os"
	"time"

	"code.justin.tv/coleiain/isengard-dev-helper/pkg/credentials"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbiface"
)

var (
	ddb          dynamodbiface.DynamoDBAPI
	tableName    = "turnip-channels"
	fileName     = "/Users/coleiain/turnip/import_testing.csv"
	awsAccountID = "170679240904" // twitchturnip+stg@amazon.com
	//awsAccountID = "573919923217" // twitchturnip+prod@amazon.com
	awsRole   = "Admin"
	awsRegion = "us-west-2"
)

func init() {
	credentials, err := credentials.GetIsengardCredentials(awsAccountID, awsRole)

	if err != nil {
		panic(err)
	}

	sess := session.New(&aws.Config{
		Credentials: credentials,
		Region:      &awsRegion,
	})

	ddb = dynamodb.New(sess)
}

func main() {
	file, err := os.Open(fileName)
	if err != nil {
		log.Fatal(err)
	}

	r := csv.NewReader(file)

	records, err := r.ReadAll()
	if err != nil {
		log.Fatal(err)
	}

	batch := make([][]string, 0, 25)

	for _, record := range records[1:] {
		if len(batch) == 25 {
			writeBatch(batch)
			batch = make([][]string, 0, 25)
			batch = append(batch, record)
		} else {
			batch = append(batch, record)
		}
	}

	if len(batch) > 0 {
		writeBatch(batch)
	}
}

type ddbChannel struct {
	ChannelIDHash string    `json:"channel_id_hash"`
	ChannelID     string    `json:"channel_id"`
	AddedBy       string    `json:"added_by"`
	AddedAt       time.Time `json:"added_at"`
	Notes         string    `json:"notes"`
}

func writeBatch(batch [][]string) {
	requests := []*dynamodb.WriteRequest{}
	for _, record := range batch {
		// Use channel if there are no notes
		notes := record[1]
		if record[2] != "" {
			notes = record[2]
		}

		item, err := dynamodbattribute.MarshalMap(&ddbChannel{
			ChannelIDHash: fmt.Sprintf("%x", sha256.Sum256([]byte(record[0]))),
			ChannelID:     record[0],
			AddedBy:       "migration",
			AddedAt:       time.Now(),
			Notes:         notes,
		})

		if err != nil {
			log.Fatal(err)
		}

		requests = append(requests, &dynamodb.WriteRequest{
			PutRequest: &dynamodb.PutRequest{
				Item: item,
			},
		})
	}

	log.Printf("Adding batch with length %v", len(requests))

	_, err := ddb.BatchWriteItem(&dynamodb.BatchWriteItemInput{
		RequestItems: map[string][]*dynamodb.WriteRequest{
			tableName: requests,
		},
	})

	if err != nil {
		log.Fatal(err)
	}
}
