package svc

import (
	"context"
	"crypto/sha256"
	"fmt"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"
	"github.com/golang/protobuf/ptypes"
	"github.com/twitchtv/twirp"

	rpc "code.justin.tv/event-engineering/turnip/pkg/rpc"
)

type ddbChannel struct {
	ChannelIDHash string    `json:"channel_id_hash"`
	ChannelID     string    `json:"channel_id"`
	AddedBy       string    `json:"added_by"`
	AddedAt       time.Time `json:"added_at"`
	Notes         string    `json:"notes"`
}

func (c *client) AddChannel(ctx context.Context, req *rpc.AddChannelRequest) (*rpc.AddChannelResponse, error) {
	// Since this is an admin function we need to figure out who the user is
	userID := UserID(ctx)
	if userID == "" {
		return nil, twirp.NewError(twirp.Malformed, fmt.Sprintf("Please provide a user identifier in the header %v", UserIDHeader))
	}

	// And make sure they're allowed to do it via BRASS
	hasPermission, err := c.hasPermission(ctx, c.isAdminBindleLockID)
	if err != nil {
		c.logger.WithError(err).Error("Failed to establish permissions")
		return nil, twirp.InternalError("Failed to establish permissions")
	}

	if !hasPermission {
		return nil, twirp.NewError(twirp.PermissionDenied, "You do not have permission to perform this action")
	}

	if req.ChannelId == "" {
		c.logger.Warn("Request to create channel with missing ChannelId")
		return nil, twirp.RequiredArgumentError("ChannelId")
	}

	// I checked with VX (the ultimate consumer of this service) and there's no
	// reason to validate that the user ID that was passed in is actually a real Twitch user ID,
	// if the IDs are invalid then they're just going to simply do nothing with it
	channelIDHash := fmt.Sprintf("%x", sha256.Sum256([]byte(req.ChannelId)))

	// This is what we're gonna return if everything checks out
	ret := &rpc.Channel{
		ChannelId: req.ChannelId,
		AddedBy:   userID,
		AddedAt:   ptypes.TimestampNow(),
		Notes:     req.Notes,
	}

	// WARNING: Fate Temptation
	// I mean cmon, can this EVER error? We don't need to do error handling here surely????
	addedAt, _ := ptypes.Timestamp(ret.AddedAt)

	ddbc := &ddbChannel{
		ChannelIDHash: channelIDHash,
		ChannelID:     ret.ChannelId,
		AddedBy:       ret.AddedBy,
		AddedAt:       addedAt,
		Notes:         ret.Notes,
	}

	item, err := dynamodbattribute.MarshalMap(ddbc)

	if err != nil {
		c.logger.WithError(err).Warnf("Failed to marshal ddbChannel")
		return nil, twirp.InternalError("Failed to create channel")
	}

	_, err = c.ddb.PutItem(&dynamodb.PutItemInput{
		TableName:           aws.String(c.channelsTableName),
		Item:                item,
		ConditionExpression: aws.String("attribute_not_exists(channel_id_hash)"),
	})

	if err != nil {
		c.logger.WithError(err).Warn("Failed to write channel to dynamodb")

		if _, ok := err.(*dynamodb.ConditionalCheckFailedException); ok {
			return nil, twirp.NewError(twirp.FailedPrecondition, "The specified channel already exists")
		}

		return nil, twirp.InternalError("Failed to create channel")
	}

	return &rpc.AddChannelResponse{
		Channel: ret,
	}, nil
}
