package svc

import (
	"context"
	"crypto/sha256"
	"fmt"

	rpc "code.justin.tv/event-engineering/turnip/pkg/rpc"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"
	"github.com/golang/protobuf/ptypes"
	"github.com/twitchtv/twirp"
)

func (c *client) GetChannel(ctx context.Context, req *rpc.GetChannelRequest) (*rpc.GetChannelResponse, error) {
	// Since this is a restricted function we need to figure out who the user is
	userID := UserID(ctx)
	if userID == "" {
		return nil, twirp.NewError(twirp.Malformed, fmt.Sprintf("Please provide a user identifier in the header %v", UserIDHeader))
	}

	// And make sure they're allowed to do it via BRASS
	hasPermission, err := c.hasPermission(ctx, c.canAccessSystemBindleLockID)
	if err != nil {
		c.logger.WithError(err).Error("Failed to establish permissions")
		return nil, twirp.InternalError("Failed to establish permissions")
	}

	if !hasPermission {
		return nil, twirp.NewError(twirp.PermissionDenied, "You do not have permission to perform this action")
	}

	if req.ChannelId == "" {
		c.logger.Warn("Request to create channel with missing ChannelId")
		return nil, twirp.RequiredArgumentError("ChannelId")
	}

	item, err := c.ddbGetChannel(fmt.Sprintf("%x", sha256.Sum256([]byte(req.ChannelId))))

	if err != nil {
		c.logger.WithError(err).Warnf("Failed to retrieve channel [%v] from dynamodb", req.ChannelId)
		return nil, twirp.InternalError("Failed to retrieve channel")
	}

	if len(item) == 0 {
		return nil, twirp.NotFoundError("Channel does not exist")
	}

	channel, err := c.hydrateChannel(item)

	if err != nil {
		c.logger.WithError(err).Warnf("Failed to hydrate channel")
		return nil, twirp.InternalError("Failed to hydrate channel")
	}

	return &rpc.GetChannelResponse{
		Channel: channel,
	}, nil
}

func (c *client) ddbGetChannel(channelIDHash string) (map[string]*dynamodb.AttributeValue, error) {
	idAttr, err := dynamodbattribute.Marshal(channelIDHash)
	if err != nil {
		return nil, err
	}

	resp, err := c.ddb.GetItem(&dynamodb.GetItemInput{
		TableName: aws.String(c.channelsTableName),
		Key: map[string]*dynamodb.AttributeValue{
			"channel_id_hash": idAttr,
		},
	})

	if err != nil {
		return nil, err
	}

	return resp.Item, nil
}

func (c *client) hydrateChannel(item map[string]*dynamodb.AttributeValue) (*rpc.Channel, error) {
	var ddbc ddbChannel
	err := dynamodbattribute.ConvertFromMap(item, &ddbc)

	if err != nil {
		return nil, err
	}

	addedAt, err := ptypes.TimestampProto(ddbc.AddedAt)

	if err != nil {
		return nil, err
	}

	return &rpc.Channel{
		ChannelId: ddbc.ChannelID,
		AddedBy:   ddbc.AddedBy,
		AddedAt:   addedAt,
		Notes:     ddbc.Notes,
	}, nil
}
