package svc

import (
	"context"
	"fmt"

	rpc "code.justin.tv/event-engineering/turnip/pkg/rpc"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/twitchtv/twirp"
)

func (c *client) ListChannels(ctx context.Context, req *rpc.ListChannelsRequest) (*rpc.ListChannelsResponse, error) {
	// Since this is a restricted function we need to figure out who the user is
	userID := UserID(ctx)
	if userID == "" {
		return nil, twirp.NewError(twirp.Malformed, fmt.Sprintf("Please provide a user identifier in the header %v", UserIDHeader))
	}

	// And make sure they're allowed to do it via BRASS
	hasPermission, err := c.hasPermission(ctx, c.canAccessSystemBindleLockID)
	if err != nil {
		c.logger.WithError(err).Error("Failed to establish permissions")
		return nil, twirp.InternalError("Failed to establish permissions")
	}

	if !hasPermission {
		return nil, twirp.NewError(twirp.PermissionDenied, "You do not have permission to perform this action")
	}

	scanInput := &dynamodb.ScanInput{
		TableName: aws.String(c.channelsTableName),
	}

	if req.NextToken != "" {
		scanInput.ExclusiveStartKey = map[string]*dynamodb.AttributeValue{
			"channel_id_hash": &dynamodb.AttributeValue{
				S: aws.String(req.NextToken),
			},
		}
	}

	resp, err := c.ddb.Scan(scanInput)

	if err != nil {
		c.logger.WithError(err).Error("Failed to query dynamodb")
		return nil, err
	}

	result := &rpc.ListChannelsResponse{
		Channels: make([]*rpc.Channel, 0, len(resp.Items)),
	}

	if resp.LastEvaluatedKey != nil {
		result.NextToken = *resp.LastEvaluatedKey["channel_id_hash"].S
	}

	for _, ch := range resp.Items {
		channel, err := c.hydrateChannel(ch)
		if err != nil {
			c.logger.WithError(err).Error("Failed to unmarshal channel from list")
			continue
		}

		result.Channels = append(result.Channels, channel)
	}

	return result, nil
}
