package svc

import (
	"context"
	"crypto/sha256"
	"fmt"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"
	"github.com/twitchtv/twirp"

	rpc "code.justin.tv/event-engineering/turnip/pkg/rpc"
)

func (c *client) RemoveChannel(ctx context.Context, req *rpc.RemoveChannelRequest) (*rpc.RemoveChannelResponse, error) {
	// Since this is an admin function we need to figure out who the user is
	userID := UserID(ctx)
	if userID == "" {
		return nil, twirp.NewError(twirp.Malformed, fmt.Sprintf("Please provide a user identifier in the header %v", UserIDHeader))
	}

	// And make sure they're allowed to do it via BRASS
	hasPermission, err := c.hasPermission(ctx, c.isAdminBindleLockID)
	if err != nil {
		c.logger.WithError(err).Error("Failed to establish permissions")
		return nil, twirp.InternalError("Failed to establish permissions")
	}

	if !hasPermission {
		return nil, twirp.NewError(twirp.PermissionDenied, "You do not have permission to perform this action")
	}

	if req.ChannelId == "" {
		c.logger.Warn("Request to remove channel with missing ChannelId")
		return nil, twirp.RequiredArgumentError("ChannelId")
	}

	idAttr, err := dynamodbattribute.Marshal(fmt.Sprintf("%x", sha256.Sum256([]byte(req.ChannelId))))
	if err != nil {
		c.logger.WithError(err).Warnf("Failed to generate ID attribute")
		return nil, twirp.InternalError("Failed to delete channel")
	}

	_, err = c.ddb.DeleteItem(&dynamodb.DeleteItemInput{
		TableName: aws.String(c.channelsTableName),
		Key:       map[string]*dynamodb.AttributeValue{"channel_id_hash": idAttr},
	})

	if err != nil {
		c.logger.WithError(err).Warn("Failed to delete channel from dynamodb")
		return nil, twirp.InternalError("Failed to delete channel")
	}

	return &rpc.RemoveChannelResponse{}, nil
}
