package svc

import (
	"context"
	"crypto/sha256"
	"fmt"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"
	"github.com/twitchtv/twirp"

	rpc "code.justin.tv/event-engineering/turnip/pkg/rpc"
)

func (c *client) UpdateChannel(ctx context.Context, req *rpc.UpdateChannelRequest) (*rpc.UpdateChannelResponse, error) {
	// Since this is an admin function we need to figure out who the user is
	userID := UserID(ctx)
	if userID == "" {
		return nil, twirp.NewError(twirp.Malformed, fmt.Sprintf("Please provide a user identifier in the header %v", UserIDHeader))
	}

	// And make sure they're allowed to do it via BRASS
	hasPermission, err := c.hasPermission(ctx, c.isAdminBindleLockID)
	if err != nil {
		c.logger.WithError(err).Error("Failed to establish permissions")
		return nil, twirp.InternalError("Failed to establish permissions")
	}

	if !hasPermission {
		return nil, twirp.NewError(twirp.PermissionDenied, "You do not have permission to perform this action")
	}

	if req.ChannelId == "" {
		c.logger.Warn("Request to update channel with missing ChannelId")
		return nil, twirp.RequiredArgumentError("ChannelId")
	}

	channelIDHash := fmt.Sprintf("%x", sha256.Sum256([]byte(req.ChannelId)))

	idAttr, err := dynamodbattribute.Marshal(channelIDHash)
	if err != nil {
		c.logger.WithError(err).Warn("Failed to update channel notes: Marshal ID")
		return nil, twirp.InternalError("Failed to update channel")
	}

	notesAttr, err := dynamodbattribute.Marshal(req.Notes)
	if err != nil {
		c.logger.WithError(err).Warn("Failed to update channel notes: Marshal Notes")
		return nil, twirp.InternalError("Failed to update channel")
	}

	_, err = c.ddb.UpdateItem(&dynamodb.UpdateItemInput{
		TableName: aws.String(c.channelsTableName),
		Key: map[string]*dynamodb.AttributeValue{
			"channel_id_hash": idAttr,
		},
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":notes":           notesAttr,
			":channel_id_hash": idAttr,
		},
		UpdateExpression:    aws.String("SET notes=:notes"),
		ConditionExpression: aws.String("channel_id_hash=:channel_id_hash"),
	})

	if err != nil {
		c.logger.WithError(err).Warn("Failed to update channel in dynamodb")

		if _, ok := err.(*dynamodb.ConditionalCheckFailedException); ok {
			return nil, twirp.NewError(twirp.FailedPrecondition, "The specified channel doesn't exist")
		}

		return nil, twirp.InternalError("Failed to update channel")
	}

	return &rpc.UpdateChannelResponse{}, nil
}
