package eventtype

import (
	"context"
	"fmt"
	"log"

	"code.justin.tv/eventbus/admin-cli/internal/client"
	"code.justin.tv/eventbus/admin-cli/internal/environment"
	"code.justin.tv/eventbus/admin-cli/internal/prompt"
	"code.justin.tv/eventbus/controlplane/infrastructure/rpc"
	"github.com/spf13/cobra"
)

var (
	eventTypeNameDelete string
)

func init() {
	eventTypeDeleteCommand.Flags().StringVarP(&eventTypeNameDelete, "event-type", "t", "", "name of event type to delete")
	eventTypeDeleteCommand.MarkFlagRequired("event-type")
}

var eventTypeDeleteCommand = &cobra.Command{
	Use:   "delete",
	Short: "Deletes an event type",
	Run: func(cmd *cobra.Command, args []string) {
		ctx := context.Background()
		config, err := environment.Resolve()
		if err != nil {
			log.Fatal("Error resolving runtime environment: ", err.Error())
		}

		c, err := client.NewInfrastructure(config)
		if err != nil {
			log.Fatal("Error creating infrastructure client: ", err.Error())
		}

		passed, err := prompt.ConfirmationCheck("Provide the event type name you are removing", "delete", "event type", []string{eventTypeNameDelete})
		if err != nil {
			log.Fatalf("Failed attempting a confirmation check: %v", err)
		} else if !passed {
			log.Fatal("Aborting due to confirmation failure")
		}

		resp, err := c.DeleteEventType(ctx, &rpc.DeleteEventTypeReq{EventType: eventTypeNameDelete})
		if err != nil {
			log.Fatal("Failed to delete event type: ", err.Error())
		}

		if resp.Deleted {
			fmt.Printf("Successfully removed %s!\n", eventTypeNameDelete)
		} else {
			fmt.Printf("Something bizarre happened... removal of %s unsuccessful (w/o error)", eventTypeNameDelete)
		}
	},
}
