package publication

import (
	"context"
	"log"

	"code.justin.tv/eventbus/admin-cli/internal/output"

	"code.justin.tv/eventbus/admin-cli/internal/client"
	"code.justin.tv/eventbus/admin-cli/internal/environment"
	"code.justin.tv/eventbus/controlplane/infrastructure/rpc"
	"github.com/spf13/cobra"
)

var (
	env, eventType string
)

func init() {
	publicationListCommand.Flags().StringVarP(&env, "environment", "e", "", "environment of publication")
	publicationListCommand.Flags().StringVarP(&eventType, "event-type", "t", "", "the name of the event being published to")
	publicationListCommand.MarkFlagRequired("environment")
	publicationListCommand.MarkFlagRequired("event-type")
}

type PublicationResults struct {
	Publications []*rpc.Publication
}

func (r PublicationResults) AsTableData() [][]string {
	data := [][]string{}
	for _, pub := range r.Publications {
		deprecated := "false"
		if pub.EventDeprecated {
			deprecated = "true"
		}
		row := []string{pub.PublisherServiceName, pub.PublisherServiceCatalogId, pub.ServiceId, pub.PublisherIamRole, deprecated, pub.SnsTopicArn, pub.EventType, pub.Environment}
		data = append(data, row)
	}
	return data
}

var publicationListCommand = &cobra.Command{
	Use:   "list",
	Short: "Lists existing publications (currently only supports listing by event stream)",
	Run: func(cmd *cobra.Command, args []string) {
		ctx := context.Background()
		config, err := environment.Resolve()
		if err != nil {
			log.Fatal("Error resolving runtime environment: ", err.Error())
		}

		c, err := client.NewInfrastructure(config)
		if err != nil {
			log.Fatal("Error creating infrastructure client: ", err.Error())
		}

		resp, err := c.ListPublicationsByEventStream(ctx, &rpc.ListPublicationsByEventStreamReq{
			Environment: env,
			EventType:   eventType,
		})
		if err != nil {
			log.Fatal("Error fetching publications by event stream: ", err.Error())
		}

		if err := output.Table([]string{"service", "service_catalog_id", "service_id", "publisher_iam_role", "deprecated", "sns_topic_arn", "event_type", "environment"}, PublicationResults{resp.GetPublications()}); err != nil {
			output.FatalError(err)
		}
		output.Newline()
	},
}
