package eventbus_test

import (
	"context"
	"errors"
	"testing"
	"time"

	"github.com/gofrs/uuid"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"

	eventbus "code.justin.tv/eventbus/client"
	"code.justin.tv/eventbus/client/internal/testschema/clock"
	"code.justin.tv/eventbus/client/internal/testschema/user_password"
	"code.justin.tv/eventbus/client/internal/wire"
)

func TestDispatchNoop(t *testing.T) {
	dispatcher := eventbus.NewMux().Dispatcher()

	buf, err := wire.DefaultEncodeNoop("development")
	require.NoError(t, err)

	t.Run("Dispatcher does not error on no-op", func(t *testing.T) {
		err := dispatcher.Dispatch(context.Background(), buf)
		require.NoError(t, err)
	})
}

func TestDispatchNoopCustom(t *testing.T) {
	mux := eventbus.NewMux()

	var returnedError error
	var calls []eventbus.RawMessage
	mux.HandleNoop(func(ctx context.Context, msg eventbus.RawMessage) error {
		calls = append(calls, msg)
		return returnedError
	})
	dispatcher := mux.Dispatcher()

	buf, err := wire.DefaultEncodeNoop("development")
	require.NoError(t, err)

	t.Run("Custom no-op handler works", func(t *testing.T) {
		returnedError = nil
		err := dispatcher.Dispatch(context.Background(), buf)
		require.NoError(t, err)
		assert.Len(t, calls, 1)
	})

	t.Run("Custom no-op handler can expose errors", func(t *testing.T) {
		returnedError = errors.New("An Error")
		err := dispatcher.Dispatch(context.Background(), buf)
		require.Error(t, err)
		require.Equal(t, "An Error", err.Error())
		assert.Len(t, calls, 2)
	})
}

func TestDispatchDefault(t *testing.T) {
	mux := eventbus.NewMux()

	user_password.RegisterUpdateHandler(mux, func(context.Context, *eventbus.Header, *user_password.Update) error {
		return nil
	})

	msgid := uuid.Must(uuid.NewV4())
	ts := time.Date(2019, 7, 1, 12, 0, 0, 0, time.UTC)
	h, _ := wire.MakeHeader(msgid, "UserPasswordUpdate", "development", ts)

	user_pw := mustEncode(wire.EncodeRaw(h, &user_password.Update{}))
	clock := mustEncode(wire.DefaultsEncode(&clock.ClockUpdate{}, "development"))

	t.Run("Default default handler", func(t *testing.T) {
		defaultDispatch := mux.Dispatcher()

		err := defaultDispatch.Dispatch(context.Background(), clock)
		require.Error(t, err)
		require.Contains(t, err.Error(), "could not find handler for ClockUpdate")

		err = defaultDispatch.Dispatch(context.Background(), user_pw)
		require.NoError(t, err)
	})

	t.Run("Custom default handler", func(t *testing.T) {
		var calls []eventbus.RawMessage
		mux.HandleDefault(func(ctx context.Context, msg eventbus.RawMessage) error {
			calls = append(calls, msg)
			require.Equal(t, "development", msg.Header.Environment)
			return nil
		})

		nfDispatch := mux.Dispatcher()
		err := nfDispatch.Dispatch(context.Background(), clock)
		require.NoError(t, err)
		require.Len(t, calls, 1)

		err = nfDispatch.Dispatch(context.Background(), user_pw)
		require.NoError(t, err)
		require.Len(t, calls, 1) // no additional calls

	})
}

func mustEncode(buf []byte, err error) []byte {
	if err != nil {
		panic(err)
	}
	return buf
}
