package mock

import (
	"errors"
	"testing"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestAuthorizedFieldClientMock(t *testing.T) {
	t.Run("No custom handler", func(t *testing.T) {
		c := &AuthorizedFieldClient{}

		s, err := c.DecryptString([]byte("foobar"))
		require.NoError(t, err)
		assert.Equal(t, "foobar", s)
		require.Len(t, c.DecryptCalls, 1)
		assert.Equal(t, []byte("foobar"), c.DecryptCalls[0].Payload)
	})

	t.Run("Custom handler", func(t *testing.T) {

		t.Run("No error", func(t *testing.T) {
			c := &AuthorizedFieldClient{}
			c.DecryptHandler = func(client *AuthorizedFieldClient, blob []byte) (string, error) {
				return "always blue", nil
			}

			s, err := c.DecryptString([]byte("foobar"))
			require.NoError(t, err)
			assert.Equal(t, "always blue", s)
			require.Len(t, c.DecryptCalls, 1)
			assert.Equal(t, []byte("foobar"), c.DecryptCalls[0].Payload)
		})

		t.Run("Forced error", func(t *testing.T) {
			c := &AuthorizedFieldClient{}
			c.DecryptHandler = func(client *AuthorizedFieldClient, blob []byte) (string, error) {
				return "", errors.New("womp womp")
			}

			s, err := c.DecryptString([]byte("foobar"))
			require.Error(t, err)
			assert.Equal(t, "", s)
			require.Len(t, c.DecryptCalls, 1)
			assert.Equal(t, []byte("foobar"), c.DecryptCalls[0].Payload)
		})

	})
}
