package publisher

import (
	"testing"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

// We're testing with an event type we know the config to exist.
func TestProdRouting(t *testing.T) {
	fetcher := DefaultRouteFetcher(routeBaseURI)
	t.Run("Existing Real Event", func(t *testing.T) {
		route, err := fetcher.FetchRoute(EnvProduction, "ClockTick")
		assert.NoError(t, err)
		require.NotNil(t, route)
		assert.NotEmpty(t, route.Arn)
	})

	t.Run("Non-existent Event", func(t *testing.T) {
		route, err := fetcher.FetchRoute(EnvProduction, "EventThatDoesntExist")
		assert.Error(t, err)
		assert.Nil(t, route)
		assert.Contains(t, err.Error(), "Got code 403 retrieving config from")
	})
}

func TestValidEventName(t *testing.T) {
	t.Run("Invalid event name, empty", func(t *testing.T) {
		assert.False(t, false, validEventName(""))
	})

	t.Run("Invalid event names, bad characters", func(t *testing.T) {
		assert.False(t, false, validEventName("myCoolEvent.bad___format"))
		assert.False(t, false, validEventName("!@#$%^&"))
		assert.False(t, false, validEventName("/"))
		assert.False(t, false, validEventName("the next!@#bus!@#to bikini!@#bottom!@#!@#!@#!@#!@#"))
		assert.False(t, false, validEventName("ewww, glove flavor..."))
	})

	t.Run("Valid event names", func(t *testing.T) {
		assert.True(t, true, validEventName("FooBar"))
		assert.True(t, true, validEventName("Foobar"))
		assert.True(t, true, validEventName("StreamUp"))
		assert.True(t, true, validEventName("a"))
		assert.True(t, true, validEventName("ItDoesntMatterHowLongTheseNamesAreWhenValidatingTheirName"))
		assert.True(t, true, validEventName("THEYCANEVENBEENTIRELYALLCAPSBUTTHATSPRETTYSILLYTOBEHONEST"))
	})
}
