locals {
  dnsmasq_cache_size = 8192
  dnsmasq_memory     = 512
  dnsmasq_ulimit     = 100000
  dnsmasq_version    = "2.78"
}

resource "aws_ecs_service" "dns" {
  name            = "dns"
  cluster         = "${aws_ecs_cluster.main.id}"
  task_definition = "${aws_ecs_task_definition.dns.arn}"

  # The daemon scheduling strategy deploys exactly one task on each active container instance.
  # This ensures that every host is running a DNS container alongside a GraphQL container.
  # https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs_services.html#service_scheduler_daemon
  scheduling_strategy = "DAEMON"
}

resource "aws_ecs_task_definition" "dns" {
  family             = "dns"
  network_mode       = "host"
  execution_role_arn = "${aws_iam_role.ecs_task_execution_role.arn}"

  container_definitions = <<DEFINITION
[
  {
    "name": "dnsmasq",
    "image": "andyshinn/dnsmasq:${local.dnsmasq_version}",
    "memory": ${local.dnsmasq_memory},
    "essential": true,
    "portMappings": [
      {
        "containerPort": 53,
        "hostPort": 53,
        "protocol": "tcp"
      },
      {
        "containerPort": 53,
        "hostPort": 53,
        "protocol": "udp"
      }
    ],
    "ulimits": [
      {
        "name": "nofile",
        "softLimit": ${local.dnsmasq_ulimit},
        "hardLimit": ${local.dnsmasq_ulimit}
      }
    ],
    "linuxParameters": {
      "capabilities": {
        "add": ["NET_ADMIN"]
      }
    },
    "command": ${jsonencode(
      concat(
        list(
          "-c", "${local.dnsmasq_cache_size}",
          "-S", "169.254.169.253"
        ),
        split(" ", join(" ", formatlist("-S /justin.tv/twitch.tv/%s", var.dns_servers))))
    )}
  }
]
DEFINITION
}
