module "twitch_networking" {
  source = "../twitch_networking"
  environment = var.environment
}

data "aws_iam_account_alias" "current" {}
data "aws_caller_identity" "current" {}

locals {
  acct_alias = "${data.aws_iam_account_alias.current.account_alias}"
  ldap_host  = "ldap.twitch.a2z.com"
}

resource "aws_alb" "main" {
  name            = "${var.cluster_name}"
  internal        = true
  subnets         = module.twitch_networking.private_subnets
  security_groups = ["${module.twitch_networking.twitch_subnets_sg_id}"]
}

resource "aws_alb_target_group" "main" {
  deregistration_delay = 5                                    # seconds
  slow_start           = "${var.lb_slow_start_seconds}"
  name                 = "${var.cluster_name}"
  protocol             = "HTTP"
  vpc_id               = "${module.twitch_networking.vpc_id}"

  # The port is required, but overriden by ECS. Using a placeholder here.
  port = "1"

  health_check {
    protocol            = "HTTP"
    path                = "/health"
    healthy_threshold   = 3         # checks resulting in success
    unhealthy_threshold = 2         # checks resulting in failure
    timeout             = 2         # seconds
    interval            = 5         # seconds
    matcher             = 200       # status code
  }
}

# Redirect all traffic from the ALB to the target group
resource "aws_alb_listener" "main" {
  load_balancer_arn = "${aws_alb.main.id}"
  port              = "80"
  protocol          = "HTTP"

  default_action {
    target_group_arn = "${aws_alb_target_group.main.id}"
    type             = "forward"
  }
}

resource "aws_alb_listener" "https" {
  load_balancer_arn = "${aws_alb.main.id}"
  port              = "443"
  protocol          = "HTTPS"
  certificate_arn   = "${aws_acm_certificate.ssl_cert.arn}"

  default_action {
    target_group_arn = "${aws_alb_target_group.main.id}"
    type             = "forward"
  }
}

# Request an SSL certificate through ACM.
# To validate a certificate, create the CNAME DNS record required by the DNS validation method
# in the internal DNS dashboard. See https://wiki.twitch.com/x/CoFQB.
resource "aws_acm_certificate" "ssl_cert" {
  domain_name       = "eventbus-clock-${var.environment}.internal.justin.tv"
  validation_method = "DNS"

  tags = {
    app = "${var.app_name}"
    env = "${var.environment}"
  }

  lifecycle {
    create_before_destroy = true
  }
}

resource "aws_iam_role_policy_attachment" "allow_send_access_logs_policy" {
  role       = "${var.ecs_host_iam_id}"
  policy_arn = "arn:aws:iam::aws:policy/AWSOpsWorksCloudWatchLogs"
}

resource "aws_cloudwatch_log_group" "app-logs" {
  name              = "${var.cluster_name}-logs"
  retention_in_days = "${var.app_log_retention_days}"

  tags = {
    app = "${var.app_name}"
    env = "${var.environment}"
  }
}

resource "aws_cloudwatch_log_group" "ecs_logs" {
  name              = "${var.cluster_name}-ecs-logs"
  retention_in_days = var.app_log_retention_days

  tags = {
    cluster = var.cluster_name
    env     = var.environment
  }
}

resource "aws_cloudwatch_log_group" "ssm-logs" {
  name              = "${var.cluster_name}-ssm-ec2-logs"
  retention_in_days = "${var.app_log_retention_days}"

  tags = {
    app     = "${var.app_name}"
    cluster = "${var.cluster_name}"
    env     = "${var.environment}"
  }
}

resource "aws_ecs_cluster" "main" {
  name = "${var.cluster_name}"
}

resource "aws_iam_role" "ecs_task_execution_role" {
  name = "ecs_task_execution_${var.environment}"

  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Sid": "",
      "Effect": "Allow",
      "Principal": {
        "Service": "ecs-tasks.amazonaws.com"
      },
      "Action": "sts:AssumeRole"
    }
  ]
}
EOF
}

resource "aws_iam_role_policy_attachment" "ecs_task_execution_role" {
  role       = "${aws_iam_role.ecs_task_execution_role.id}"
  policy_arn = "arn:aws:iam::aws:policy/service-role/AmazonECSTaskExecutionRolePolicy"
}

resource "aws_iam_role" "ecs_task_role" {
  name = "ecs_task_role_${var.environment}"

  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Sid": "",
      "Effect": "Allow",
      "Principal": {
        "Service": "ecs-tasks.amazonaws.com"
      },
      "Action": "sts:AssumeRole"
    }
  ]
}
EOF
}

resource "aws_iam_role_policy_attachment" "s3_role" {
  role       = "${aws_iam_role.ecs_task_role.id}"
  policy_arn = "arn:aws:iam::aws:policy/AmazonS3FullAccess"
}

resource "aws_ecs_task_definition" "app" {
  family             = "${var.cluster_name}"
  network_mode       = "bridge"
  execution_role_arn = "${aws_iam_role.ecs_task_execution_role.arn}"
  task_role_arn      = "${aws_iam_role.ecs_task_role.arn}"

  container_definitions = <<DEFINITION
[
  {
    "cpu": ${var.task_cpu},
    "image": "${var.app_image}",
    "memory": ${var.task_memory},
    "name": "${var.cluster_name}",
    "environment": [
      {
        "name": "ENVIRONMENT",
        "value": "${var.environment}"
      }
    ],
    "essential": true,
    "logConfiguration": {
      "logDriver": "awslogs",
      "options": {
        "awslogs-group": "${aws_cloudwatch_log_group.app-logs.name}",
        "awslogs-region": "${var.region}",
        "awslogs-stream-prefix": "${var.cluster_name}"
      }
    },
    "placementConstraints": [
      {
        "type": "distinctInstance"
      }
    ],
    "portMappings": [
      {
        "containerPort": ${var.app_port},
        "protocol": "tcp"
      }
    ],
    "dnsServers": ["172.17.0.1"],
    "ulimits": [{
      "name": "nofile",
      "softLimit": 65535,
      "hardLimit": 65535
    }]
  }
]
DEFINITION
}

# Define how our upcoming ECS cluster will get autoscaled & how the cluster's EC2's are deployed and configured
# Note: installs LDAP-AUTH on the ec2's pointing to our special LDAP hosts
module "ecs_asg" {
  source = "git::git+ssh://git@git-aws.internal.justin.tv/twitch/terraform//ecsasg?ref=v2.1.1"

  cluster_name                  = "${var.cluster_name}"
  task_cleanup_wait_duration    = "${var.task_cleanup_wait_duration}"
  container_instance_profile_id = "${var.ecs_host_iam_id}"
  desired_capacity              = "${var.cluster_size["desired"]}"
  max_size                      = "${var.cluster_size["max"]}"
  min_size                      = "${var.cluster_size["min"]}"
  environment                   = "${var.environment}"
  instance_type                 = "${var.ec2_instance_type}"
  ldap_group                    = "${var.ssh_sudoer_ldap_group}"
  custom_ldap_host              = "${local.ldap_host}"
  name                          = "${var.cluster_name}"
  owner                         = "${var.owner_email}"

  ecs_init_loggroup  = aws_cloudwatch_log_group.ecs_logs.name
  ecs_agent_loggroup = aws_cloudwatch_log_group.ecs_logs.name

  ssm_agent_loggroup = "${aws_cloudwatch_log_group.ssm-logs.name}"
  team               = "${var.team_name}"

  security_groups = "${module.twitch_networking.twitch_subnets_sg_id}"

  # Subnets to which ASG will add new ec2's. Yeah the var name is weird
  vpc_zone_identifier = "${join(",", module.twitch_networking.private_subnets)}"

  # How do we scale up in case of high cpu?
  scale_out_cooldown           = 1200
  scale_out_evaluation_periods = 5
  scale_out_increment          = 1

  # set really high - let's avoid unnecessary scaling up
  scale_out_cpu_percent_threshold = 90

  # How do we scale down in case of low cpu?
  scale_in_cooldown           = 900
  scale_in_evaluation_periods = 15
  scale_in_increment          = -1

  # set really low - let's avoid unnecessary scaling down
  scale_in_cpu_percent_threshold = 0.01
}

resource "aws_ecs_service" "main" {
  name            = "${var.cluster_name}"
  cluster         = "${aws_ecs_cluster.main.id}"
  task_definition = "${aws_ecs_task_definition.app.arn}"
  desired_count   = "${var.app_count}"
  iam_role        = "${var.service_iam_role}"

  lifecycle {
    ignore_changes = ["task_definition"]
  }

  load_balancer {
    target_group_arn = "${coalesce(var.target_group_id, aws_alb_target_group.main.id)}"
    container_name   = "${var.cluster_name}"
    container_port   = "${var.app_port}"
  }

  depends_on = [
    "aws_alb_listener.main",
    "aws_alb_listener.https",
  ]
}

resource "aws_iam_service_linked_role" "ecs" {
  aws_service_name = "ecs.amazonaws.com"
}

resource "aws_iam_role_policy" "sns_policy" {
  name   = "sns_policy"
  policy = "${data.aws_iam_policy_document.sns_policy.json}"
  role   = "${aws_iam_role.ecs_task_role.name}"
}

data "aws_iam_policy_document" "sns_policy" {
  statement {
    actions   = ["sns:*"]
    effect    = "Allow"
    resources = ["*"]
  }
}
