package main

import (
	"context"
	"os"
	"os/signal"
	"syscall"
	"time"

	telemetry "code.justin.tv/amzn/TwitchTelemetry"
	"code.justin.tv/eventbus/controlplane/infrastructure"
	"code.justin.tv/eventbus/controlplane/infrastructure/notification"
	"code.justin.tv/eventbus/controlplane/infrastructure/validation"

	"code.justin.tv/eventbus/controlplane/internal/autoprof"
	"code.justin.tv/eventbus/controlplane/internal/clients/kms"
	"code.justin.tv/eventbus/controlplane/internal/clients/servicecatalog"
	"code.justin.tv/eventbus/controlplane/internal/clients/slack"
	"code.justin.tv/eventbus/controlplane/internal/clients/sns"
	"code.justin.tv/eventbus/controlplane/internal/clients/sqs"
	"code.justin.tv/eventbus/controlplane/internal/clients/sts"
	"code.justin.tv/eventbus/controlplane/internal/db"
	"code.justin.tv/eventbus/controlplane/internal/db/observability"
	"code.justin.tv/eventbus/controlplane/internal/db/postgres"
	"code.justin.tv/eventbus/controlplane/internal/environment"
	"code.justin.tv/eventbus/controlplane/internal/metrics"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/s3"

	"code.justin.tv/eventbus/controlplane/internal/logger"
	"go.uber.org/zap"
)

const (
	// How long to wait between validation runs
	defaultValidationInterval = 5 * time.Minute

	// How long to allow an in-progress validation to keep running after stop signal before shutting it down
	validationGracePeriod = 10 * time.Second
)

func main() {
	ctx := telemetry.ContextWithOperationName(context.Background(), "validator")
	log := logger.FromContext(ctx)
	ctx = log.AddToContext(ctx)
	serviceCatalog := servicecatalog.New()

	config, err := environment.Read()
	if err != nil {
		log.Fatal("could not initialize config", zap.Error(err))
	}

	sess := session.Must(session.NewSession(config.AWSConfig()))

	if config.CloudwatchEnabled {
		if err := metrics.InitializeProcessIdentifier("eventbus-validator"); err != nil {
			log.Fatal("error initializing process identifier", zap.Error(err))
		}
		metrics.InitializeMetrics(log.Logger)
		metrics.StartGoStatsCollection(log.Logger)
	}

	if config.AutoprofEnabled {
		bucket := config.AutoprofBucketName
		if bucket == "" {
			log.Fatal("autoprof enabled but no bucket provided")
		}
		autoprof.Start(s3.New(sess), bucket, log)
	}

	dbConn, err := dbConn(config)
	if err != nil {
		log.Fatal("could not create db connection", zap.Error(err))
	}

	authFieldGrantManager := kms.NewManager(sess, config.AuthorizedFieldKeyARN)

	slackClient, err := slackClient(config)
	if err != nil {
		log.Fatal("could not create slack client", zap.Error(err))
	}

	notificationChannels := []notification.Channel{
		notification.NewSlack(slackClient, time.Hour),
	}

	if config.CloudwatchEnabled {
		notificationChannels = append(notificationChannels, notification.NewCloudwatch(metrics.ProcessIdentifier(), log))
	}

	if config.PagerDutyEnabled {
		low := config.PagerDutyLowUrgencyKey
		high := config.PagerDutyHighUrgencyKey
		notificationChannels = append(notificationChannels, notification.NewPagerDuty(low, high))
	}

	validationInterval := environment.Duration("VALIDATION_INTERVAL", defaultValidationInterval)

	// Allow for semi-graceful shutdown mid-validation
	gracefulStopCtx, gracefulStopCancel := context.WithCancel(ctx)
	defer gracefulStopCancel()

	// validationCtx, validationCancel := context.WithCancel(loggerCtx)
	hardCancelCtx, hardCancel := context.WithCancel(ctx)
	defer hardCancel()

	listenForSignal(gracefulStopCancel, hardCancel)

	stsManager := sts.NewManager(sess, config.Environment)
	sqsManager := sqs.NewManager(sess, stsManager)
	snsManager := sns.NewManager(sess, stsManager, config.EncryptionAtRestKeyARN, log)

	eventBusAccountID, err := stsManager.GetBaseAWSAccountID(ctx)
	if err != nil {
		log.Fatal("Could not determine aws account id", zap.Error(err))
	}

	validationConfig := validation.Config{
		DB:                     dbConn,
		ServiceCatalogClient:   serviceCatalog,
		GrantFetcher:           authFieldGrantManager,
		SQSManager:             sqsManager,
		SNSManager:             snsManager,
		EventBusAWSAccountID:   eventBusAccountID,
		EncryptionAtRestKeyARN: config.EncryptionAtRestKeyARN,
	}
	for {
		items, err := validation.CollectItems(hardCancelCtx, sess, validationConfig)
		if err != nil {
			log.Error("could not collect items to validate", zap.Error(err))
		}

		err = infrastructure.Validate(hardCancelCtx, gracefulStopCtx, items, notificationChannels)
		if err != nil {
			log.Error("error running validator", zap.Error(err))
		}

		select {
		case <-gracefulStopCtx.Done():
			return
		case <-time.After(validationInterval):
		}
	}
}

// tries to cancel gracefully, but will call an ungraceful cancel if
// enough time elapses and the application is still running
func listenForSignal(gracefulCancel, ungracefulCancel func()) {
	sighandler := make(chan os.Signal, 1)

	signal.Notify(sighandler, os.Interrupt, syscall.SIGTERM)

	go func() {
		for range sighandler {
			gracefulCancel()
			time.AfterFunc(validationGracePeriod, ungracefulCancel)
		}
	}()
}

func dbConn(config environment.Config) (db.DB, error) {
	var dbConn db.DB
	var err error
	sslMode := "require"
	if config.PostgresDisableSSL {
		sslMode = "disable"
	}
	dbConn, err = postgres.New(postgres.Config{
		Reader: postgres.ConnectionConfig{
			Username: config.PostgresReaderUsername,
			Password: config.PostgresReaderPassword,
			Hostname: config.PostgresReaderHostname,
			Dbname:   config.PostgresDBName,
			Sslmode:  sslMode,
		},
		Writer: postgres.ConnectionConfig{
			Username: config.PostgresWriterUsername,
			Password: config.PostgresWriterPassword,
			Hostname: config.PostgresWriterHostname,
			Dbname:   config.PostgresDBName,
			Sslmode:  sslMode,
		},
	})
	if err != nil {
		return nil, err
	}

	dbConn = observability.WithLogging(dbConn)
	if config.CloudwatchEnabled {
		dbConn = observability.WithMetrics(dbConn)
	}

	return dbConn, nil
}

func slackClient(config environment.Config) (slack.Slack, error) {
	var slackClient slack.Slack
	if config.SlackChannel != "" {
		var err error
		slackClient, err = slack.New(&slack.Config{
			Token:   config.SlackToken,
			Channel: config.SlackChannel,
		})
		if err != nil {
			return nil, err
		}
	} else {
		slackClient = &slack.Noop{}
	}

	return slackClient, nil
}
