package main

import (
	"context"

	"code.justin.tv/eventbus/controlplane/e2e/internal/suite"
	"code.justin.tv/eventbus/controlplane/e2e/internal/test"
	"code.justin.tv/eventbus/controlplane/internal/e2eaccounts"
	"code.justin.tv/eventbus/controlplane/internal/logger"
	"github.com/aws/aws-sdk-go/aws/session"
	"go.uber.org/zap"
)

var baseSession *session.Session // access to the main e2e test account

var log *logger.Logger
var ctx context.Context

// Use init to setup basic credentials with access to each of the E2E test accounts
func init() {
	ctx = context.Background()
	log = logger.FromContext(ctx)

	baseSession = session.Must(session.NewSession())

	log.Info("Generating all required account credentials")
	err := e2eaccounts.InitializeCredentials(baseSession)
	if err != nil {
		log.Fatal("Could not get credentials for all required E2E test AWS accounts", zap.Error(err))
	}
}

func main() {
	log.Info("EventBus End-to-End Test Suite")

	cfg, err := test.ConfigFromEnvironment()
	if err != nil {
		log.Fatal("Could not determine runtime configuration", zap.Error(err))
	}

	err = e2eaccounts.InitializeCredentials(baseSession)
	if err != nil {
		log.Fatal("Could not initialize AWS credentials")
	}

	type Entry struct {
		Path    string
		Builder func(string) (suite.Runner, error)
	}

	entries := []Entry{
		{
			Path:    "01_basic",
			Builder: suite.NewBasicTestSuite,
		},
		{
			Path:    "02_clients",
			Builder: suite.NewClientsTestSuite,
		},
		{
			Path:    "03_unencrypted",
			Builder: suite.NewUnencryptedTestSuite,
		},
		{
			Path:    "04_resubscribe",
			Builder: suite.NewResubscribeTestSuite,
		},
		{
			Path:    "07_event_pubsub_information",
			Builder: suite.NewEventPubsubInformation,
		},
		{
			Path:    "09_authorized_field_grants",
			Builder: suite.NewAuthorizedFieldGrantsTestSuite,
		},
		{
			Path:    "10_iam_role_publication",
			Builder: suite.NewBasicTestSuite,
		},
		{
			Path:    "11_feature_flags",
			Builder: suite.NewFeatureFlagTestSuite,
		},
		{
			Path:    "12_iam_role_delete",
			Builder: suite.NewIAMRoleDeleteTestSuite,
		},
		{
			Path:    "13_queue_validation",
			Builder: suite.NewQueueValidationTestSuite,
		},
		{
			Path:    "14_topic_validation",
			Builder: suite.NewTopicValidationTestSuite,
		},
		{
			Path:    "15_subscription_validation",
			Builder: suite.NewSubscriptionValidationTestSuite,
		},
		{
			Path:    "16_authorized_field_validation",
			Builder: suite.NewAuthorizedFieldValidationTestSuite,
		},
		{
			Path:    "17_publication_validation",
			Builder: suite.NewPublicationValidationTestSuite,
		},
		{
			Path:    "18_create_iam_role_api",
			Builder: suite.NewCreateIAMRoleAPI,
		},
		{
			Path:    "19_event_type_ownership",
			Builder: suite.NewEventTypeOwnershipTestSuite,
		},
	}

	for _, entry := range entries {
		runner, err := entry.Builder(entry.Path)
		if err != nil {
			log.Fatal("Could not prepare test", zap.String("testName", entry.Path), zap.Error(err))
		}
		suite.Run(ctx, runner, cfg)
	}

	log.Info("All tests have passed successfully! Amazing! Stupendous! Marvelous! Yeehaw!")
}
