package e2eutil

import (
	"context"
	"os"
	"strings"

	"code.justin.tv/eventbus/controlplane/e2e/internal/expected"
	"code.justin.tv/eventbus/controlplane/internal/clients/kms"
	"code.justin.tv/eventbus/controlplane/internal/clients/servicecatalog"
	"code.justin.tv/eventbus/controlplane/internal/clients/sns"
	"code.justin.tv/eventbus/controlplane/internal/clients/sqs"
	"code.justin.tv/eventbus/controlplane/internal/clients/sts"
	"code.justin.tv/eventbus/controlplane/internal/db"
	"code.justin.tv/eventbus/controlplane/internal/db/postgres"
	"code.justin.tv/eventbus/controlplane/internal/e2eaccounts"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/pkg/errors"

	"code.justin.tv/eventbus/controlplane/infrastructure"

	"code.justin.tv/eventbus/controlplane/infrastructure/notification"
	"code.justin.tv/eventbus/controlplane/infrastructure/validation"
)

func Validate(ctx context.Context) ([]*validation.Report, error) {
	sess, err := session.NewSession(e2eaccounts.AccountCredentials().MainConfig())
	if err != nil {
		return nil, errors.Wrap(err, "could not initialize session")
	}

	psqlConf := postgres.Config{
		Reader: postgres.ConnectionConfig{
			Username: os.Getenv("POSTGRES_READER_USER"),
			Password: os.Getenv("POSTGRES_READER_PASSWORD"),
			Hostname: os.Getenv("POSTGRES_READER_HOST"),
			Dbname:   os.Getenv("POSTGRES_DB"),
			Sslmode:  "disable",
		},
		Writer: postgres.ConnectionConfig{
			Username: os.Getenv("POSTGRES_WRITER_USER"),
			Password: os.Getenv("POSTGRES_WRITER_PASSWORD"),
			Hostname: os.Getenv("POSTGRES_WRITER_HOST"),
			Dbname:   os.Getenv("POSTGRES_DB"),
			Sslmode:  "disable",
		},
	}

	var dbConn db.DB
	dbConn, err = postgres.New(psqlConf)
	if err != nil {
		return nil, errors.Wrap(err, "could not connect to db")
	}

	authFieldGrantFetcher := kms.NewManager(sess, expected.AuthorizedFieldKeyARN)

	sc := servicecatalog.New()

	stsManager := sts.NewManager(sess, "e2e-test")
	sqsManager := sqs.NewManager(sess, stsManager)
	snsManager := sns.NewManager(sess, stsManager, expected.AuthorizedFieldKeyARN, nil)

	validationConfig := validation.Config{
		DB:                     dbConn,
		ServiceCatalogClient:   sc,
		GrantFetcher:           authFieldGrantFetcher,
		SQSManager:             sqsManager,
		SNSManager:             snsManager,
		EventBusAWSAccountID:   expected.BaseAWSAccountID,
		EncryptionAtRestKeyARN: expected.EncryptionAtRestKeyARN,
	}
	items, err := validation.CollectItems(ctx, sess, validationConfig)
	if err != nil {
		return nil, errors.Wrap(err, "could not collect items to validate")
	}

	filteredItems := make([]validation.Item, 0)
	for _, item := range items {
		if strings.Contains(item.ID(), JobID(ctx)) {
			filteredItems = append(filteredItems, item)
		}
	}
	return ValidateWithItems(ctx, filteredItems)
}

func ValidateWithItems(ctx context.Context, items []validation.Item) ([]*validation.Report, error) {
	collector := newReportCollector()
	err := infrastructure.Validate(ctx, ctx, items, []notification.Channel{collector})
	if err != nil {
		return nil, errors.Wrap(err, "could not run validation routine")
	}

	return collector.Reports(), nil
}

func ErrorsFor(name string, reports []*validation.Report) bool {
	for _, report := range reports {
		if strings.Contains(report.Item.ID(), name) {
			return true
		}
	}
	return false
}

func ReportsFor(name string, reports []*validation.Report) []*validation.Report {
	matchingReports := make([]*validation.Report, 0)
	for _, report := range reports {
		if strings.Contains(report.Item.ID(), name) {
			matchingReports = append(matchingReports, report)
		}
	}
	return matchingReports
}

// ReportCollector gathers validation reports
type reportCollector struct {
	reports []*validation.Report
}

func newReportCollector() *reportCollector {
	return &reportCollector{
		reports: make([]*validation.Report, 0),
	}
}

func (r *reportCollector) Add(report *validation.Report) {
	r.reports = append(r.reports, report)
}

func (r *reportCollector) Submit(ctx context.Context) error {
	return nil
}

func (r *reportCollector) Reset() {
	r.reports = make([]*validation.Report, 0)
}

func (r *reportCollector) Reports() []*validation.Report {
	return r.reports
}
