package resource

import (
	"context"
	"fmt"
	"strings"

	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/sns"
	"github.com/pkg/errors"

	"code.justin.tv/eventbus/controlplane/e2e/internal/e2eutil"
	"code.justin.tv/eventbus/controlplane/e2e/internal/expected"
	"code.justin.tv/eventbus/controlplane/e2e/internal/httpserver"
	"code.justin.tv/eventbus/controlplane/e2e/internal/report"
	"code.justin.tv/eventbus/controlplane/e2e/internal/test"
	"code.justin.tv/eventbus/controlplane/internal/e2eaccounts"
	"code.justin.tv/eventbus/controlplane/rpc"
)

const awsRetries = 5

var _ test.Runner = &Subscription{}

var ErrSubscriptionNotFound = errors.New("could not find subscription corresponding to provided target")

type Subscription struct {
	test.TestRunner

	EventType        string `json:"event_type"`
	Environment      string `json:"environment"`
	TargetName       string `json:"target_name"`
	ServiceCatalogID string `json:"service_catalog_id"`

	skipClean bool
}

func (s *Subscription) AWSSubscription(ctx context.Context) (*sns.Subscription, error) {
	sess := session.Must(session.NewSession())
	snsClient := sns.New(sess, e2eaccounts.AccountCredentials().SubscriberConfig())

	var foundSubscription *sns.Subscription

	// passed into the ListSubscriptionPages function, cycles through subscriptions
	// to find the target. Will indicate to stop paging when either there are no more
	// pages or the sub has been found
	subFinder := func(subsPage *sns.ListSubscriptionsOutput, lastPage bool) bool {
		partialTargetName := e2eutil.Suffix(s.TargetName, e2eutil.JobID(ctx)[:10])
		for _, awsSub := range subsPage.Subscriptions {
			awsSubTargetName, err := expected.TargetFromSubscriptionEndpointARN(*awsSub.Endpoint)
			if err != nil {
				return false
			}
			if strings.Contains(awsSubTargetName, partialTargetName) {
				foundSubscription = awsSub
				break
			}
		}
		// we only want to keep paging if we havent found the sub AND there are more pages
		return foundSubscription == nil && !lastPage
	}

	// TODO [ASYNC-487] - audit other e2e AWS calls to determine if they need pagination as well
	err := snsClient.ListSubscriptionsPagesWithContext(ctx, &sns.ListSubscriptionsInput{}, subFinder)
	if err != nil {
		return nil, err
	}

	// Look only for part of the jobID - subscription names may be truncated due to max character length

	if foundSubscription == nil {
		return nil, ErrSubscriptionNotFound
	}
	return foundSubscription, nil
}

func (s *Subscription) AWSSubscriptionPoll(ctx context.Context) (*sns.Subscription, error) {
	var awsSub *sns.Subscription

	err := e2eutil.RetryBackoff("aws subscription poll", awsRetries, func() (err error) {
		s.Log(ctx, "Polling for aws subscription")
		awsSub, err = s.AWSSubscription(ctx)
		return err
	})
	return awsSub, err
}

func (s *Subscription) Clean(ctx context.Context) []report.Error {
	if s.skipClean {
		return nil
	}

	ctx = e2eutil.AppendTestPath(ctx, s.TestName())
	sub, err := s.AWSSubscriptionPoll(ctx)
	if err != nil {
		return reportErrorSlice(ctx, "Could not get subscription", err)
	}

	sess := session.Must(session.NewSession())
	snsClient := sns.New(sess, e2eaccounts.AccountCredentials().SubscriberConfig())
	_, err = snsClient.UnsubscribeWithContext(context.Background(), &sns.UnsubscribeInput{
		SubscriptionArn: sub.SubscriptionArn,
	})
	if err != nil {
		return reportErrorSlice(ctx, "Could not unsubscribe queue from topic", err)
	}
	return nil
}

func (s *Subscription) TestName() string {
	return fmt.Sprintf("Subscription{%s,%s,%s,%s}", s.EventType, s.Environment, s.TargetName, s.ServiceCatalogID)
}

func (s *Subscription) Setup(ctx context.Context) report.Error {
	ctx = e2eutil.AppendTestPath(ctx, s.TestName())
	s.Log(ctx, "Creating subscription")

	foundTarget, err := httpserver.Target(s.ServiceCatalogID, s.TargetName, e2eutil.JobID(ctx))
	if err != nil {
		return report.ErrorFromContext(ctx, "Could not find subscription target for subscription", err)
	}

	httpClient := e2eutil.HTTPClientWithLDAP()
	subscriptionsClient := rpc.NewSubscriptionsJSONClient(expected.TwirpEndpoint, httpClient)
	_, err = subscriptionsClient.Create(ctx, &rpc.CreateSubscriptionReq{
		TargetId:    foundTarget.Id,
		EventType:   e2eutil.Suffix(s.EventType, e2eutil.JobID(ctx)),
		Environment: s.Environment,
	})
	if err != nil {
		return report.ErrorFromContext(ctx, "Could not create subscription in database", err)
	}
	return nil
}

func (s *Subscription) Test(ctx context.Context) {
	ctx = e2eutil.AppendTestPath(ctx, s.TestName())
	// inject a bit of context into our logger
	s.Log(ctx, "Beginning subscriptions tests")

	// base checks
	s.subscriptionExistenceCheck(ctx)

	// vvv optional tests are conditionally run here, if specified in the seed data vvv

}

func (s *Subscription) subscriptionExistenceCheck(ctx context.Context) {

	ctx = e2eutil.AppendTestPath(ctx, s.TestName())
	s.Log(ctx, "Checking that subscription exists")
	_, err := s.AWSSubscriptionPoll(ctx)
	if err != nil {
		s.Error(ctx, "Could not verify existence of AWS subscription", err)
	}
}

func (s *Subscription) SkipClean(b bool) {
	s.skipClean = b
}
