package servicecatalog

import (
	"context"
	"testing"

	"github.com/stretchr/testify/require"
)

type mockFetcher struct {
	mockServices []Service
}

func (f *mockFetcher) fetchServices(ctx context.Context) ([]Service, error) {
	return f.mockServices, nil
}

func TestPopulateCache(t *testing.T) {
	ctx := context.Background()
	t.Run("should populate the cache with results from service catalog", func(t *testing.T) {
		mockServices := []Service{
			Service{
				ID:   "1",
				Name: "serv1",
			},
			Service{
				ID:   "2",
				Name: "serv2",
			},
		}
		m := &mockFetcher{
			mockServices: mockServices,
		}
		s := New()
		s.fetcher = m
		err := s.populateCache(ctx)
		require.NoError(t, err)

		item := s.cache.Get("1").Value().(*Response)
		require.Equal(t, item.Name, "serv1")

		item = s.cache.Get("2").Value().(*Response)
		require.Equal(t, item.Name, "serv2")
	})

	t.Run("should override the results of previous caches", func(t *testing.T) {
		mockServices := []Service{
			Service{
				ID:   "1",
				Name: "serv1",
			},
		}
		m := &mockFetcher{
			mockServices,
		}
		s := New()
		s.fetcher = m
		err := s.populateCache(ctx)
		require.NoError(t, err)
		item := s.cache.Get("1").Value().(*Response)
		require.Equal(t, item.Name, "serv1")

		m.mockServices = []Service{
			Service{
				ID:   "1",
				Name: "overriden name",
			},
		}
		err = s.populateCache(ctx)
		require.NoError(t, err)
		item = s.cache.Get("1").Value().(*Response)
		require.Equal(t, item.Name, "overriden name")
	})
}

func TestCacheMissingService(t *testing.T) {
	t.Run("should populate cache with placeholder", func(t *testing.T) {
		s := New()
		s.cacheMissingService("1")
		item := s.cache.Get("1")
		require.Equal(t, item.Value(), &NonExistentService{})
	})
}
