package observability

import (
	"context"
	"time"

	"code.justin.tv/eventbus/controlplane/internal/db"
	"go.uber.org/zap"
)

func (o *ObservabilityDB) SubscriptionTargetCreate(ctx context.Context, subscriptionTarget *db.SubscriptionTarget) (int, error) {
	var err error
	var id int
	o.observe(ctx, "SubscriptionTargetCreate",
		func() ([]zap.Field, error) {
			id, err = o.db.SubscriptionTargetCreate(ctx, subscriptionTarget)
			cpy := *subscriptionTarget
			cpy.ID = id
			return []zap.Field{
				zap.Object("subscriptionTarget", cpy),
			}, err
		})
	return id, err
}
func (o *ObservabilityDB) SubscriptionTargetUpdate(ctx context.Context, id int, subscriptionTargetEditable *db.SubscriptionTargetEditable) (int, error) {
	var err error
	var retID int
	o.observe(ctx, "SubscriptionTargetUpdate",
		func() ([]zap.Field, error) {
			retID, err = o.db.SubscriptionTargetUpdate(ctx, id, subscriptionTargetEditable)
			return []zap.Field{
				zap.Int("subscriptionTargetID", id),
				zap.Object("subscriptionTargetUpdate", subscriptionTargetEditable),
			}, err
		})
	return retID, err
}
func (o *ObservabilityDB) SubscriptionTargetUpdateInfra(ctx context.Context, lease db.AWSLease, id int, subscriptionTargetInfraUpdate *db.SubscriptionTargetInfraUpdate) (int, error) {
	var err error
	var retID int
	o.observe(ctx, "SubscriptionTargetUpdateInfra",
		func() ([]zap.Field, error) {
			retID, err = o.db.SubscriptionTargetUpdateInfra(ctx, lease, id, subscriptionTargetInfraUpdate)
			return []zap.Field{
				zap.Int("subscriptionTargetID", id),
				zap.Object("subscriptionTargetInfraUpdate", subscriptionTargetInfraUpdate),
				zap.Object("lease", lease),
			}, err
		})
	return retID, err
}
func (o *ObservabilityDB) SubscriptionTargetDelete(ctx context.Context, lease db.AWSLease, id int) error {
	var err error
	o.observe(ctx, "SubscriptionTargetDelete",
		func() ([]zap.Field, error) {
			err = o.db.SubscriptionTargetDelete(ctx, lease, id)
			return []zap.Field{
				zap.Int("subscriptionTargetID", id),
				zap.Object("lease", lease),
			}, err
		})
	return err
}
func (o *ObservabilityDB) SubscriptionTargets(ctx context.Context) ([]*db.SubscriptionTarget, error) {
	var err error
	var targets []*db.SubscriptionTarget
	o.observe(ctx, "SubscriptionTargets",
		func() ([]zap.Field, error) {
			targets, err = o.db.SubscriptionTargets(ctx)
			return []zap.Field{}, err
		})
	return targets, err
}
func (o *ObservabilityDB) SubscriptionTargetByID(ctx context.Context, id int) (*db.SubscriptionTarget, error) {
	var err error
	var target *db.SubscriptionTarget
	o.observe(ctx, "SubscriptionTargetByID",
		func() ([]zap.Field, error) {
			target, err = o.db.SubscriptionTargetByID(ctx, id)
			return []zap.Field{
				zap.Int("subscriptionTargetID", id),
			}, err
		})
	return target, err
}
func (o *ObservabilityDB) SubscriptionTargetByQueueURL(ctx context.Context, queueURL string) (*db.SubscriptionTarget, error) {
	var err error
	var target *db.SubscriptionTarget
	o.observe(ctx, "SubscriptionTargetByQueueURL",
		func() ([]zap.Field, error) {
			target, err = o.db.SubscriptionTargetByQueueURL(ctx, queueURL)
			return []zap.Field{
				zap.String("queueURL", queueURL),
			}, err
		})
	return target, err
}
func (o *ObservabilityDB) SubscriptionTargetsByServiceID(ctx context.Context, serviceID int) ([]*db.SubscriptionTarget, error) {
	var err error
	var targets []*db.SubscriptionTarget
	o.observe(ctx, "SubscriptionTargetsByServiceID",
		func() ([]zap.Field, error) {
			targets, err = o.db.SubscriptionTargetsByServiceID(ctx, serviceID)
			return []zap.Field{
				zap.Int("serviceID", serviceID),
			}, err
		})
	return targets, err
}
func (o *ObservabilityDB) SubscriptionTargetAcquireLease(ctx context.Context, resourceID int, timeout time.Duration) (db.AWSLease, context.Context, error) {
	var err error
	var awsLease db.AWSLease
	var retCtx context.Context
	o.observe(ctx, "SubscriptionTargetAcquireLease",
		func() ([]zap.Field, error) {
			var fields []zap.Field
			awsLease, retCtx, err = o.db.SubscriptionTargetAcquireLease(ctx, resourceID, timeout)
			if awsLease != nil {
				fields = append(fields, zap.Object("lease", awsLease))
			}
			return append(fields, zap.Int("subscriptionTargetID", resourceID)), err
		},
	)
	return awsLease, retCtx, err
}
func (o *ObservabilityDB) SubscriptionTargetReleaseLease(lease db.AWSLease) error {
	return o.db.SubscriptionTargetReleaseLease(lease)
}
