package observability

import (
	"context"
	"time"

	"code.justin.tv/eventbus/controlplane/internal/db"
	"go.uber.org/zap"
)

func (o *ObservabilityDB) SubscriptionCreate(ctx context.Context, subscription *db.Subscription) (int, error) {
	var err error
	var id int
	o.observe(ctx, "SubscriptionCreate",
		func() ([]zap.Field, error) {
			id, err = o.db.SubscriptionCreate(ctx, subscription)
			cpy := *subscription
			cpy.ID = id
			return []zap.Field{
				zap.Object("subscription", cpy),
			}, err
		})
	return id, err
}
func (o *ObservabilityDB) SubscriptionUpdateInfra(ctx context.Context, lease db.AWSLease, id int, subscriptionInfraUpdate *db.SubscriptionInfraUpdate) (int, error) {
	var err error
	var retID int
	o.observe(ctx, "SubscriptionUpdateInfra",
		func() ([]zap.Field, error) {
			retID, err = o.db.SubscriptionUpdateInfra(ctx, lease, id, subscriptionInfraUpdate)
			return []zap.Field{
				zap.Object("lease", lease),
				zap.Object("subscriptionInfraUpdate", subscriptionInfraUpdate),
			}, err
		})
	return retID, err
}
func (o *ObservabilityDB) SubscriptionDelete(ctx context.Context, lease db.AWSLease, id int) error {
	var err error
	o.observe(ctx, "SubscriptionDelete",
		func() ([]zap.Field, error) {
			err = o.db.SubscriptionDelete(ctx, lease, id)
			return []zap.Field{
				zap.Int("subscriptionID", id),
				zap.Object("lease", lease),
			}, err
		})
	return err
}
func (o *ObservabilityDB) Subscriptions(ctx context.Context) ([]*db.Subscription, error) {
	var err error
	var subs []*db.Subscription
	o.observe(ctx, "Subscriptions",
		func() ([]zap.Field, error) {
			subs, err = o.db.Subscriptions(ctx)
			return []zap.Field{}, err
		})
	return subs, err
}
func (o *ObservabilityDB) SubscriptionByID(ctx context.Context, id int) (*db.Subscription, error) {
	var err error
	var sub *db.Subscription
	o.observe(ctx, "SubscriptionByID",
		func() ([]zap.Field, error) {
			sub, err = o.db.SubscriptionByID(ctx, id)
			return []zap.Field{
				zap.Int("subscriptionID", id),
			}, err
		})
	return sub, err
}
func (o *ObservabilityDB) SubscriptionByEventStreamIDAndSubscriptionTargetID(ctx context.Context, eventStreamID, subscriptionTargetID int) (*db.Subscription, error) {
	var err error
	var sub *db.Subscription
	o.observe(ctx, "SubscriptionByEventStreamIDAndSubscriptionTargetID",
		func() ([]zap.Field, error) {
			sub, err = o.db.SubscriptionByEventStreamIDAndSubscriptionTargetID(ctx, eventStreamID, subscriptionTargetID)
			return []zap.Field{
				zap.Int("eventStreamID", eventStreamID),
				zap.Int("subscriptionTargetID", subscriptionTargetID),
			}, err
		})
	return sub, err
}
func (o *ObservabilityDB) SubscriptionsBySubscriptionTargetID(ctx context.Context, subscriptionTargetID int) ([]*db.Subscription, error) {
	var err error
	var subs []*db.Subscription
	o.observe(ctx, "SubscriptionsBySubscriptionTargetID",
		func() ([]zap.Field, error) {
			subs, err = o.db.SubscriptionsBySubscriptionTargetID(ctx, subscriptionTargetID)
			return []zap.Field{
				zap.Int("subscriptionTargetID", subscriptionTargetID),
			}, err
		})
	return subs, err
}
func (o *ObservabilityDB) SubscriptionsByEventStreamID(ctx context.Context, eventStreamID int) ([]*db.Subscription, error) {
	var err error
	var subs []*db.Subscription
	o.observe(ctx, "SubscriptionsByEventStreamID",
		func() ([]zap.Field, error) {
			subs, err = o.db.SubscriptionsByEventStreamID(ctx, eventStreamID)
			return []zap.Field{
				zap.Int("eventStreamID", eventStreamID),
			}, err
		})
	return subs, err
}
func (o *ObservabilityDB) SubscriptionAcquireLease(ctx context.Context, resourceID int, timeout time.Duration) (db.AWSLease, context.Context, error) {
	var err error
	var awsLease db.AWSLease
	var retCtx context.Context
	o.observe(ctx, "SubscriptionAcquireLease",
		func() ([]zap.Field, error) {
			awsLease, retCtx, err = o.db.SubscriptionAcquireLease(ctx, resourceID, timeout)
			fields := []zap.Field{
				zap.Int("subscriptionID", resourceID),
			}
			if awsLease != nil {
				fields = append(fields, zap.Object("lease", awsLease))
			}
			return fields, err
		})
	return awsLease, retCtx, err
}
func (o *ObservabilityDB) SubscriptionReleaseLease(lease db.AWSLease) error {
	return o.db.SubscriptionReleaseLease(lease)
}
