package postgres

import (
	"context"
	"database/sql"

	"code.justin.tv/eventbus/controlplane/internal/db"
	"github.com/jmoiron/sqlx"
	"github.com/pkg/errors"
)

const (
	EventTypesTableName = "event_types"

	eventTypeCreateQuery = `
INSERT INTO event_types (name, description, schema, repo_filepath, deprecated, ldap_group)
VALUES (:name, :description, :schema, :repo_filepath, FALSE, :ldap_group)
RETURNING id
`

	eventTypeAllQuery = `
SELECT * FROM event_types
ORDER BY name
`

	eventTypeWithIDQuery = `
SELECT * FROM event_types
WHERE id = $1
`
	eventTypeWithNameQuery = `
SELECT * FROM event_types
WHERE name = $1
`

	eventTypeUpdateQuery = `
UPDATE event_types
SET name = $2, description = $3, schema = $4, repo_filepath = $5, deprecated = $6, ldap_group = $7
WHERE id = $1
`

	eventTypesDeprecatedQuery = `
SELECT * FROM event_types
WHERE deprecated = true
ORDER BY name
`

	eventTypeDeleteByNameQuery = `
DELETE FROM event_types
WHERE name = $1
`
)

func (pg *PostgresDB) EventTypeCreate(ctx context.Context, eventType *db.EventType) (int, error) {
	var tx *sqlx.Tx
	var err error
	var res int
	if tx, err = pg.newTx(); err != nil {
		return -1, err
	}
	if res, err = pg.eventTypeCreateTx(ctx, tx, eventType); err != nil {
		return -1, err
	}
	if err = pg.doneTx(tx); err != nil {
		return -1, nil
	}
	return res, nil
}

func (pg *PostgresDB) eventTypeCreateTx(ctx context.Context, tx *sqlx.Tx, eventType *db.EventType) (int, error) {
	var id int
	namedQuery, err := tx.PrepareNamedContext(ctx, eventTypeCreateQuery)
	if err != nil {
		return -1, err
	}

	err = namedQuery.QueryRowxContext(ctx, eventType).Scan(&id)
	return id, err
}

func (pg *PostgresDB) EventTypeUpdate(ctx context.Context, id int, eventTypeEditable *db.EventTypeEditable) (int, error) {
	updateRes, err := pg.writer.ExecContext(ctx, eventTypeUpdateQuery, id, eventTypeEditable.Name, eventTypeEditable.Description, eventTypeEditable.Schema, eventTypeEditable.RepoFilePath, eventTypeEditable.Deprecated, eventTypeEditable.LDAPGroup)
	if err != nil {
		return -1, err
	}
	numRowsAffected, err := updateRes.RowsAffected()
	if err != nil {
		return -1, errors.Wrap(err, "could not determine rows affected")
	}
	if numRowsAffected == 0 {
		return -1, db.ErrResourceNotFound
	}

	return id, nil
}

func (pg *PostgresDB) EventTypes(ctx context.Context) ([]*db.EventType, error) {
	var eventTypes []*db.EventType
	err := pg.reader.SelectContext(ctx, &eventTypes, eventTypeAllQuery)
	return eventTypes, err
}

func (pg *PostgresDB) EventTypesDeprecated(ctx context.Context) ([]*db.EventType, error) {
	var eventTypes []*db.EventType
	err := pg.reader.SelectContext(ctx, &eventTypes, eventTypesDeprecatedQuery)
	return eventTypes, err
}

func (pg *PostgresDB) EventTypeByID(ctx context.Context, id int) (*db.EventType, error) {
	var eventType db.EventType
	err := pg.reader.GetContext(ctx, &eventType, eventTypeWithIDQuery, id)
	if err == sql.ErrNoRows {
		return nil, db.ErrResourceNotFound
	}
	return &eventType, err
}

func (pg *PostgresDB) EventTypeByName(ctx context.Context, name string) (*db.EventType, error) {
	var eventType db.EventType
	err := pg.reader.GetContext(ctx, &eventType, eventTypeWithNameQuery, name)
	if err == sql.ErrNoRows {
		return nil, pgError(err, EventTypesTableName)
	}
	return &eventType, err
}

func (pg *PostgresDB) EventTypeDeleteByName(ctx context.Context, name string) error {
	row, err := pg.writer.ExecContext(ctx, eventTypeDeleteByNameQuery, name)
	if err != nil {
		return errors.Wrap(err, "failed to delete from event types")
	}

	countDeleted, err := row.RowsAffected()
	if err != nil {
		return errors.Wrap(err, "could not count rows deleted")
	}

	if countDeleted == 0 {
		return db.ErrResourceNotFound
	}

	return nil
}
