package logger

import (
	"net/http"
	"time"

	"code.justin.tv/eventbus/controlplane/internal/ldap"
	"code.justin.tv/eventbus/controlplane/internal/uuid"
	"go.uber.org/zap"
)

type RequestLogger struct {
	http.ResponseWriter
	status int
}

func (w *RequestLogger) WriteHeader(status int) {
	w.status = status
	w.ResponseWriter.WriteHeader(status)
}

func (l *Logger) Middleware() func(http.Handler) http.Handler {
	return func(next http.Handler) http.Handler {
		return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			requestID, err := uuid.NewUUID()
			if err != nil {
				w.WriteHeader(http.StatusInternalServerError)
				_, _ = w.Write([]byte("Fatal Error: Could not generate UUID"))
				return
			}

			startTime := time.Now()
			ctx := r.Context()
			user := ldap.User(ctx)

			thisLog := l.With(zap.String("requestID", requestID)).With(zap.String("user", user))
			logContext := thisLog.AddToContext(ctx)
			r = r.WithContext(logContext)

			requestLogger := &RequestLogger{w, 200}
			next.ServeHTTP(requestLogger, r)

			thisLog.Info("completed",
				zap.Object("request", Request{
					Request:   r,
					Status:    requestLogger.status,
					StartTime: startTime,
					ID:        requestID,
				}),
			)
		})
	}
}
