package validator

import (
	"testing"

	"github.com/stretchr/testify/assert"
)

func TestIsValidIAMRole(t *testing.T) {
	tests := []struct {
		arn      string
		expected bool
	}{

		{"arn:aws:iam::123456789012:role/abc", true},
		{"arn:aws:iam::123456789012:role/hello-iam+going_to=explode,FROM.tHiS@test", true},
		{"arn:aws:iam::567312298267:role/EventBus-e2e-test/EventBus-e2e-test", true},
		{"arn:aws:iam::567312298267:role/--/++/==", true},

		{"", false},
		{"troll", false},
		{"arn:aws:iam::567312298267:role/EventBus-e2e-test/EventBus-e2e-test/*", false},
		{"arn:aws:iam::123456789012:role/*", false},
		{"arn:aws:iam::123456789012:role/EventBus*", false},
		{"arn:aws:iam:123456789012:role/ok_lets_be_reasonable", false},
		{"arn:aws::123456789012:role/ok_lets_be_reasonable", false},
		{"arn:aWs:iAm::123456789012:role/ok_lets_be_reasonable", false},
		{"arn:aws:iam::123456789012:user/legal_legends", false},
		{"arn1:aws:iam::123456789012:role/ok_lets_be_reasonable", false},
		{"arn:1aws:iam::123456789012:role/ok_lets_be_reasonable", false},
		{"arn:aws:i1am::123456789012:role/ok_lets_be_reasonable", false},
		{"arn:aws:iam::12345678901:role/ok_lets_be_reasonable", false},
		{"arn:aws:iam::123456789012:role/~", false},
		{"arn:aws:iam::123456789012:role/(", false},
		{"arn:aws:iam::123456789012:role/abc)", false},
		{"arn:aws:iam::123456789012:role", false},
		{"arn:aws:iam::123456789012:role/must/not/end/with/slash/", false},
		{"arn:aws:iam::123456789012:role/no//multi/slash", false},
		{"arn:aws:iam::123456789012:role/", false},
	}

	for _, test := range tests {
		if test.expected {
			assert.True(t, IsValidIAMRoleARN(test.arn), "expected valid iam role arn %q", test.arn)
		} else {
			assert.False(t, IsValidIAMRoleARN(test.arn), "expected invalid iam role arn %q", test.arn)
		}
	}
}

func TestIsValidIAMRoleLabel(t *testing.T) {
	tests := []struct {
		label    string
		expected bool
	}{

		{"clock-dev", true},
		{"abcdefghijklmnop", true},
		{"label-1", true},
		{"  nate loves dota", true},
		{"LeaGUe_iS-bEtt3R  ", true},

		{"", false},
		{"   ", false},
		{"abcdabcdabcdabcdabcdabcdabcdabcdabcdabcdabcdabcdabcdabcdabcdabcd", false},
		{"~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~", false},
	}

	for _, test := range tests {
		if test.expected {
			assert.True(t, IsValidIAMRoleLabel(test.label), "expected valid label %q", test.label)
		} else {
			assert.False(t, IsValidIAMRoleLabel(test.label), "expected invalid label %q", test.label)
		}
	}
}
