import { Errored } from 'dashboard/core/components/errored';
import { StyledLoadingSpinner } from 'dashboard/core/components/loading-spinner';
import { GetDashboardEndpoint } from 'dashboard/core/utils/env';
import { useLDAPGroupMembers } from 'dashboard/core/utils/ldap';
import { useServices } from 'dashboard/core/utils/service';
import { ALL_ENVIRONMENTS } from 'dashboard/definitions/environments';
import { code } from 'dashboard/generated/controlplane';
import * as _ from 'lodash';
import * as React from 'react';
import { useEffect, useState } from 'react';
import {
  Background,
  BorderRadius,
  Button,
  ButtonSize,
  CoreLink,
  CoreText,
  Display,
  FlexDirection,
  FontSize,
  JustifyContent,
  Layout,
  LoadingSpinner,
  Select,
  Table,
  TableCell,
  TableHeader,
  TableHeading,
  TableRow,
  TextAlign,
  TextType,
} from 'twitch-core-ui';

// tslint:disable-next-line
const copy = require("clipboard-copy");

import twirp = code.justin.tv.eventbus.controlplane;

interface AuthorizedFieldsTableProps {
  event: twirp.IEventStream;
}

export const AuthorizedFieldsTable = ({
  event,
}: AuthorizedFieldsTableProps) => {
  const [
    selectedField,
    setSelectedField,
  ] = useState<twirp.IAuthorizedField | null>(null);
  const handleRequestAccessClick = (field: twirp.IAuthorizedField) => () =>
    setSelectedField(field);
  if (_.isEmpty(event.authorizedFields)) {
    return (
      <CoreText type={TextType.Em}>
        No authorized fields for this event
      </CoreText>
    );
  }

  return (
    <Layout padding={1}>
      <Layout padding={{ y: 1 }} textAlign={TextAlign.Center}>
        <Table>
          <TableHeader>
            <TableHeading label="Message Name" textAlign={TextAlign.Center} />
            <TableHeading label="Field Name" textAlign={TextAlign.Center} />
            <TableHeading label="Access" textAlign={TextAlign.Center} />
          </TableHeader>
          {_.map(event.authorizedFields, (authorizedField, i) => {
            return (
              <TableRow key={i}>
                <TableCell>
                  <CoreText className="font-monospace">
                    {authorizedField.messageName}
                  </CoreText>
                </TableCell>
                <TableCell>
                  <CoreText className="font-monospace">
                    {authorizedField.fieldName}
                  </CoreText>
                </TableCell>
                <TableCell>
                  <Button
                    size={ButtonSize.Small}
                    onClick={handleRequestAccessClick(authorizedField)}
                  >
                    Request Access
                  </Button>
                </TableCell>
              </TableRow>
            );
          })}
        </Table>
      </Layout>
      <Layout padding={{ y: 1 }}>
        {selectedField && (
          <AccessRequestLinkForm
            ownerLDAPGroup={event.ldapGroup!}
            eventType={event.eventType!}
            authorizedField={selectedField}
          />
        )}
      </Layout>
    </Layout>
  );
};

interface AccessRequestLinkFormProps {
  ownerLDAPGroup: string;
  eventType: string;
  authorizedField: twirp.IAuthorizedField;
}

const AccessRequestLinkForm = ({
  ownerLDAPGroup,
  authorizedField,
  eventType,
}: AccessRequestLinkFormProps) => {
  const { services, loading, error } = useServices();

  const [selectedServiceName, setSelectedServiceName] = useState('');
  const [selectedIAMRoleARN, setSelectedIAMRoleARN] = useState('');
  const [selectedEnvironment, setSelectedEnvironment] = useState('');
  const [shouldGenerateURL, setShouldGenerateURL] = useState(false);

  // Reset the form if a different "Request Access" button is pressed
  useEffect(() => {
    setSelectedServiceName('');
    setSelectedIAMRoleARN('');
    setSelectedEnvironment('');
  }, [authorizedField]);

  const handleServiceSelection = (e: React.FormEvent<HTMLSelectElement>) => {
    setSelectedServiceName(e.currentTarget.value);
  };

  const handleIAMRoleSelection = (e: React.FormEvent<HTMLSelectElement>) => {
    setSelectedIAMRoleARN(e.currentTarget.value);
  };

  const handleEnvironmentSelection = (
    e: React.FormEvent<HTMLSelectElement>,
  ) => {
    setSelectedEnvironment(e.currentTarget.value);
  };

  const handleGenerateURL = () => setShouldGenerateURL(true);

  if (loading) {
    return (
      <Layout>
        <LoadingSpinner />
      </Layout>
    );
  } else if (error) {
    return (
      <Layout>
        <Errored text={error.message} />
      </Layout>
    );
  }

  const serviceNameOptions = _.map(services, (s) => s.name!);

  const selectedService =
    selectedServiceName === ''
      ? undefined
      : _.find(services, { name: selectedServiceName });

  const iamRoleARNOptions = selectedService
    ? _.map(selectedService.iamRoles!, (iamRole) => iamRole.arn!)
    : new Array<string>();

  const urlReady =
    selectedServiceName !== '' &&
    selectedIAMRoleARN !== '' &&
    selectedEnvironment !== '';

  return (
    <Layout padding={2} elevation={1}>
      <Layout>
        <Layout>
          <CoreText bold fontSize={FontSize.Size4}>
            Requesting Access to Authorized Field
          </CoreText>
        </Layout>

        <Layout
          display={Display.Flex}
          flexDirection={FlexDirection.Row}
          padding={{ y: 1 }}
        >
          <Layout>
            <CoreText bold fontSize={FontSize.Size5}>
              Message Name:
            </CoreText>
          </Layout>
          <Layout margin={{ x: 1 }}>
            <CoreText fontSize={FontSize.Size5}>
              {authorizedField.messageName}
            </CoreText>
          </Layout>
        </Layout>
        <Layout display={Display.Flex} flexDirection={FlexDirection.Row}>
          <Layout>
            <CoreText bold fontSize={FontSize.Size5}>
              Field Name:
            </CoreText>
          </Layout>
          <Layout margin={{ x: 1 }}>
            <CoreText fontSize={FontSize.Size5}>
              {authorizedField.fieldName}
            </CoreText>
          </Layout>
        </Layout>
      </Layout>
      <Layout margin={{ y: 1 }}>
        <CoreText bold>Service</CoreText>
        <Select value={selectedServiceName} onChange={handleServiceSelection}>
          <option key={'null'} />
          {_.map(serviceNameOptions, (name) => {
            return <option key={name}>{name}</option>;
          })}
        </Select>
      </Layout>
      <Layout margin={{ y: 1 }}>
        <CoreText bold>IAM Role</CoreText>
        <Select value={selectedIAMRoleARN} onChange={handleIAMRoleSelection}>
          <option key={'null'} />
          {_.map(iamRoleARNOptions, (arn, i) => {
            return <option key={i}>{arn}</option>;
          })}
        </Select>
      </Layout>
      <Layout margin={{ y: 1 }}>
        <CoreText bold>Environment</CoreText>
        <Select
          value={selectedEnvironment}
          onChange={handleEnvironmentSelection}
        >
          <option key={'null'} />
          {_.map(ALL_ENVIRONMENTS, (env) => {
            return <option key={env}>{env}</option>;
          })}
        </Select>
      </Layout>
      <Layout display={Display.Flex} justifyContent={JustifyContent.Center}>
        <Button disabled={!urlReady} onClick={handleGenerateURL}>
          Generate
        </Button>
      </Layout>
      {shouldGenerateURL && (
        <Layout padding={3}>
          <GeneratedAccessRequestLink
            eventType={eventType}
            ownerLDAPGroup={ownerLDAPGroup}
            iamRoleARN={selectedIAMRoleARN}
            environment={selectedEnvironment}
            messageName={authorizedField.messageName!}
            fieldName={authorizedField.fieldName!}
          />
        </Layout>
      )}
    </Layout>
  );
};

interface GeneratedAccessRequestLinkProps {
  eventType: string;
  ownerLDAPGroup: string;
  iamRoleARN: string;
  environment: string;
  messageName: string;
  fieldName: string;
}

const GeneratedAccessRequestLink = ({
  ownerLDAPGroup,
  eventType,
  iamRoleARN,
  environment,
  messageName,
  fieldName,
}: GeneratedAccessRequestLinkProps) => {
  const { ldapGroupMembers, loading, error } = useLDAPGroupMembers(
    ownerLDAPGroup,
  );

  const generatedURL = `${GetDashboardEndpoint()}/events/${encodeURIComponent(
    eventType,
  )}/manage?arn=${encodeURIComponent(iamRoleARN)}&env=${encodeURIComponent(
    environment,
  )}&msg=${encodeURIComponent(messageName)}&field=${encodeURIComponent(
    fieldName,
  )}`;

  const [showCopySuccess, setShowCopySuccess] = useState(false);
  const handleCopyToClipboard = () => {
    copy(generatedURL);
    setShowCopySuccess(true);
  };

  return (
    <Layout
      background={Background.Alt2}
      borderRadius={BorderRadius.Small}
      padding={1}
      textAlign={TextAlign.Center}
      justifyContent={JustifyContent.Center}
    >
      {loading && <StyledLoadingSpinner />}
      {/* {error && <Errored text={error.message} />} */}
      {!loading && !error && (
        <>
          <Layout margin={{ y: 1 }}>
            <Layout margin={{ x: 3 }}>
              <CoreText italic>
                To gain access to this authorized field, share the generated
                link below with one of the members of the LDAP group (
                {ownerLDAPGroup}) that owns this event.
              </CoreText>
            </Layout>
          </Layout>

          <Layout margin={{ y: 2 }}>
            <Layout>
              <CoreText bold>Request Link</CoreText>
            </Layout>
            <Layout>
              <CoreText>{generatedURL}</CoreText>
            </Layout>
            <Layout>
              <CoreLink onClick={handleCopyToClipboard}>Click to copy</CoreLink>
            </Layout>
            {showCopySuccess && (
              <Layout>
                <CoreText fontSize={FontSize.Size8}>Copied!</CoreText>
              </Layout>
            )}
          </Layout>

          <Layout margin={{ y: 1 }}>
            <Layout>
              <CoreText bold>Members of LDAP Group: {ownerLDAPGroup}</CoreText>
            </Layout>
            {_.map(ldapGroupMembers, (member) => (
              <Layout>
                <CoreText>{member}</CoreText>
              </Layout>
            ))}
          </Layout>
        </>
      )}
    </Layout>
  );
};
