import { code } from 'dashboard/generated/controlplane';
import twirp = code.justin.tv.eventbus.controlplane;
import { isDuplicateServiceCatalogID, isInvalidServiceCatalogURL, serviceFullyFilled } from './service';

it('returns true when fully filled out without IAM roles', () => {
  const service = new twirp.Service({
    serviceCatalogUrl: 'https://catalog.xarth.tv/services/126/details',
    ldapGroup: 'infra',
  });
  expect(serviceFullyFilled(service)).toBe(true);
});

it('returns true when fully filled out with IAM roles', () => {
  const service = new twirp.Service({
    serviceCatalogUrl: 'https://catalog.xarth.tv/services/126/details',
    ldapGroup: 'infra',
    iamRoles: [{
      arn: 'arn:aws:iam::111111111111:role/lots-of-ones',
      label: 'lots-of-ones',
    }, {
      arn: 'arn:aws:iam::222222222222:role/lots-of-twos',
      label: 'lots-of-twos',
    }],
  });
  expect(serviceFullyFilled(service)).toBe(true);
});

it('returns false when missing service fields', () => {
  const service = new twirp.Service({
    serviceCatalogUrl: 'https://catalog.xarth.tv/services/126/details',
    // no ldap group provided
  });
  expect(serviceFullyFilled(service)).toBe(false);
});

it('returns false when missing IAM role fields', () => {
  const service = new twirp.Service({
    serviceCatalogUrl: 'https://catalog.xarth.tv/services/126/details',
    ldapGroup: 'infra',
    iamRoles: [{
      arn: 'arn:aws:iam::111111111111:role/lots-of-ones',
      // no label provided
    }],
  });
  expect(serviceFullyFilled(service)).toBe(false);
});

it('returns false when the service catalog ID is unique', () => {
  const service = new twirp.Service({
    serviceCatalogUrl: 'https://catalog.xarth.tv/services/126/details',
    ldapGroup: 'infra',
    iamRoles: [{
      arn: 'arn:aws:iam::111111111111:role/lots-of-ones',
      label: 'lots-of-ones',
    }, {
      arn: 'arn:aws:iam::222222222222:role/lots-of-twos',
      label: 'lots-of-twos',
    }],
  });
  expect(isDuplicateServiceCatalogID(service, ['123', '124', '125'])).toBe(false);
});

it('returns true when the service catalog ID is not unique', () => {
  const service = new twirp.Service({
    serviceCatalogUrl: 'https://catalog.xarth.tv/services/126/details',
    ldapGroup: 'infra',
    iamRoles: [{
      arn: 'arn:aws:iam::111111111111:role/lots-of-ones',
      label: 'lots-of-ones',
    }, {
      arn: 'arn:aws:iam::222222222222:role/lots-of-twos',
      label: 'lots-of-twos',
    }],
  });
  expect(isDuplicateServiceCatalogID(service, ['123', '124', '125', '126'])).toBe(true);
});

it('returns false for valid service catalog URLs', () => {
  expect(isInvalidServiceCatalogURL('https://catalog.xarth.tv/services/1/details')).toBe(false);
  expect(isInvalidServiceCatalogURL('https://catalog.xarth.tv/services/11/details')).toBe(false);
  expect(isInvalidServiceCatalogURL('https://catalog.xarth.tv/services/111111/details')).toBe(false);
  expect(isInvalidServiceCatalogURL('https://catalog.xarth.tv/services/2/details')).toBe(false);
  expect(isInvalidServiceCatalogURL('https://catalog.xarth.tv/services/22/details')).toBe(false);
  expect(isInvalidServiceCatalogURL('https://catalog.xarth.tv/services/222222/details')).toBe(false);
});

it('returns true for invalid service catalog URLs', () => {
  expect(isInvalidServiceCatalogURL('definitely-not-right')).toBe(true);
  expect(isInvalidServiceCatalogURL('123')).toBe(true);
  expect(isInvalidServiceCatalogURL('catalog.xarth.tv/services/123/details')).toBe(true);
  expect(isInvalidServiceCatalogURL('httpdd://catalog.xarth.tv/services/123/details')).toBe(true);
  expect(isInvalidServiceCatalogURL('https://catalog.xarth.tv/services/126/details/abc')).toBe(true);
});
