import { Errored } from 'dashboard/core/components/errored';
import { StyledLoadingSpinner } from 'dashboard/core/components/loading-spinner';
import { ResourceLoadStatus } from 'dashboard/definitions/resources';
import { code } from 'dashboard/generated/controlplane';
import * as _ from 'lodash';
import * as React from 'react';
import { AlignItems, Button, ButtonSize, ButtonType, Color, CoreText, Display, JustifyContent, Layout, Table, TableBody, TableCell, TableHeader, TableHeading, TableRow, TextType } from 'twitch-core-ui';
import { COLUMN_NAMES, displayFunctions, filter, filterableFields, filterAuditLogs } from './util';

import twirp = code.justin.tv.eventbus.controlplane;

interface Props {
  auditLogs: twirp.AuditLog[];
  filters: filter[];
  loading: ResourceLoadStatus;
  openDetailsModal: (i: number) => void;
}

export class AuditLogsTable extends React.PureComponent<Props> {
  public render() {
    const filteredAuditLogs = filterAuditLogs(this.props.auditLogs, this.props.filters, filterableFields);
    const rows = _.map(filteredAuditLogs, (auditLog, i) => {
      return (
        <TableRow key={i}>
          <TableCell>{displayFunctions.timestamp(auditLog)}</TableCell>
          <TableCell>{displayFunctions.userName(auditLog)}</TableCell>
          <TableCell>{displayFunctions.resourceType(auditLog)}</TableCell>
          <TableCell>{displayFunctions.resourceName(auditLog)}</TableCell>
          <TableCell>{displayFunctions.action(auditLog)}</TableCell>
          <TableCell>{displayFunctions.result(auditLog)}</TableCell>
          <TableCell>
            <Button
              type={ButtonType.Secondary}
              size={ButtonSize.Small}
              onClick={this.props.openDetailsModal.bind(this, i)}
            >
              View Details
            </Button>
          </TableCell>
        </TableRow>
      );
    });

    if (_.isEmpty(rows)) {
      return (
        <Layout padding={1} alignItems={AlignItems.Center}>
          <CoreText type={TextType.H4} color={Color.Alt2}>This service does not have any corresponding audit logs.</CoreText>
        </Layout>
      );
    }

    return (
      <Layout
        display={Display.Flex}
        justifyContent={JustifyContent.Center}
      >
        {this.props.loading === ResourceLoadStatus.LOADING && <StyledLoadingSpinner/>}
        {this.props.loading === ResourceLoadStatus.ERRORED && <Errored text={'Error fetching audit logs...'}/>}
        {this.props.loading === ResourceLoadStatus.COMPLETED &&
        <Table>
          <TableHeader>
            <TableHeading label={COLUMN_NAMES.timestamp}/>
            <TableHeading label={COLUMN_NAMES.userName}/>
            <TableHeading label={COLUMN_NAMES.resourceType}/>
            <TableHeading label={COLUMN_NAMES.resourceName}/>
            <TableHeading label={COLUMN_NAMES.action}/>
            <TableHeading label={COLUMN_NAMES.result}/>
            <TableHeading label={'Details'}/>
          </TableHeader>
          <TableBody>
            {rows}
          </TableBody>
        </Table>
        }
      </Layout>
    );
  }
}
