import { CustomModal } from 'dashboard/core/components/modal';
import { ModalContext } from 'dashboard/core/contexts/modals';
import { updateIAMRoleLabel } from 'dashboard/core/utils/iam_role';
import { code } from 'dashboard/generated/controlplane';
import * as React from 'react';
import { useContext, useState } from 'react';
import {
  Button,
  ButtonType,
  Color,
  Column,
  CoreText,
  Display,
  FormGroup,
  FormGroupOrientation,
  Grid,
  Input,
  InputType,
  JustifyContent,
  Layout,
} from 'twitch-core-ui';

import twirp = code.justin.tv.eventbus.controlplane;

const PLACEHOLDER_IAM_ROLE_LABEL =
  'Alias for your IAM role (e.g. myservice-prod)';

interface Props {
  iamRole: twirp.IIAMRole;
  onSuccess: (iamRole: twirp.IIAMRole) => void;
}

export const EditIAMRoleLabelModal = ({ onSuccess, iamRole }: Props) => {
  const [label, setLabel] = useState(iamRole.label ? iamRole.label : '');
  const [errorText, setErrorText] = useState('');
  const { hideModal } = useContext(ModalContext);

  const handleLabelChange = (
    e: React.FormEvent<
      HTMLInputElement | HTMLTextAreaElement | HTMLSelectElement
    >,
  ) => {
    setLabel(e.currentTarget.value);
  };

  const handleIAMRoleLabelUpdate = () => {
    updateIAMRoleLabel(
      {
        arn: iamRole.arn,
        label,
      },
      (updatedIAMRole: twirp.IIAMRole) => {
        onSuccess(updatedIAMRole);
        hideModal();
      },
      (e: Error) => setErrorText(e.message),
    );
  };

  const submitDisabled = label === '' || label === iamRole.label;

  return (
    <CustomModal
      title={`Update label for IAM role`}
      isOpen={true}
      onClose={hideModal}
    >
      <Grid>
        <Column cols={12}>
          <Layout padding={1}>
            <FormGroup
              label="ARN"
              id="arn"
              orientation={FormGroupOrientation.Horizontal}
            >
              <Input
                type={InputType.Text}
                id="arn"
                value={iamRole.arn as string}
                disabled={true}
              />
            </FormGroup>
          </Layout>
        </Column>
        <Column cols={12}>
          <Layout padding={1}>
            <FormGroup
              label="Label"
              id="label"
              orientation={FormGroupOrientation.Horizontal}
            >
              <Input
                type={InputType.Text}
                placeholder={PLACEHOLDER_IAM_ROLE_LABEL}
                id="label"
                value={label}
                onChange={handleLabelChange}
              />
            </FormGroup>
          </Layout>
        </Column>
        {errorText && (
          <Column cols={12}>
            <CoreText color={Color.Error}>{errorText}</CoreText>
          </Column>
        )}
        <Column cols={12}>
          <Layout
            padding={1}
            display={Display.Flex}
            justifyContent={JustifyContent.End}
          >
            <Button
              type={ButtonType.Success}
              onClick={handleIAMRoleLabelUpdate}
              disabled={submitDisabled}
            >
              Update
            </Button>
          </Layout>
        </Column>
      </Grid>
    </CustomModal>
  );
};
