import { CustomModal } from 'dashboard/core/components/modal';
import { ModalContext } from 'dashboard/core/contexts/modals';
import { registerTarget } from 'dashboard/core/utils/target';
import { code } from 'dashboard/generated/controlplane';
import * as React from 'react';
import { useContext, useState } from 'react';
import {
  Button,
  ButtonSize,
  ButtonType,
  Color,
  CoreText,
  Display,
  FormGroup,
  FormGroupOrientation,
  Input,
  InputType,
  JustifyContent,
  Layout,
} from 'twitch-core-ui';
import twirp = code.justin.tv.eventbus.controlplane;

interface Props {
  service: twirp.IService;
  onSuccess: (_: twirp.ITarget) => void;
}

export const RegisterTargetModal = ({ service, onSuccess }: Props) => {
  const { hideModal } = useContext(ModalContext);

  const [error, setError] = useState<Error | null>(null);
  const onError = (e: Error) => setError(e);

  const [name, setName] = useState('');
  const handleNameChange = (
    e: React.FormEvent<
      HTMLInputElement | HTMLTextAreaElement | HTMLSelectElement
    >,
  ) => setName(e.currentTarget.value);

  const [queueURL, setQueueURL] = useState('');
  const handleQueueURLChange = (
    e: React.FormEvent<
      HTMLInputElement | HTMLTextAreaElement | HTMLSelectElement
    >,
  ) => setQueueURL(e.currentTarget.value);

  const submitEnabled = name !== '' && queueURL !== '';

  const handleRegisterTarget = () => {
    registerTarget(
      {
        serviceId: service.id,
        name,
        sqs: {
          queueUrl: queueURL,
        },
      },
      (t: twirp.ITarget) => {
        onSuccess(t);
        hideModal();
      },
      onError,
    );
  };

  return (
    <CustomModal
      title={`Register Subscription Target`}
      isOpen={true}
      onClose={hideModal}
      styleOverrides={{ content: { minWidth: '70rem' } }}
    >
      <Layout margin={{ y: 1 }}>
        <FormGroup
          required
          id="name"
          label="Name"
          orientation={FormGroupOrientation.Horizontal}
        >
          <Input id="name" type={InputType.Text} onChange={handleNameChange} />
        </FormGroup>
      </Layout>
      <Layout margin={{ y: 1 }}>
        <FormGroup
          required
          id="queueUrl"
          label="SQS Queue URL"
          orientation={FormGroupOrientation.Horizontal}
        >
          <Input
            id="queueUrl"
            type={InputType.Text}
            onChange={handleQueueURLChange}
          />
        </FormGroup>
      </Layout>
      <Layout display={Display.Flex} justifyContent={JustifyContent.Between}>
        <Layout>
          {error && <CoreText color={Color.Error}>{error.message}</CoreText>}
        </Layout>
        <Layout display={Display.Flex} justifyContent={JustifyContent.Between}>
          <Layout>
            <Button
              disabled={!submitEnabled}
              type={ButtonType.Success}
              size={ButtonSize.Large}
              onClick={handleRegisterTarget}
            >
              Register Subscription Target
            </Button>
          </Layout>
        </Layout>
      </Layout>
    </CustomModal>
  );
};
