import { Errored } from 'dashboard/core/components/errored';
import { Forbidden } from 'dashboard/core/components/forbidden';
import { StyledLoadingSpinner } from 'dashboard/core/components/loading-spinner';
import { NotFound } from 'dashboard/core/components/not-found';
import { ModalContext } from 'dashboard/core/contexts/modals';
import { useService } from 'dashboard/core/utils/service';
import { EditServiceModal } from 'dashboard/pages/service-overview/components/edit-service-modal';
import { ServiceOverviewTabs } from 'dashboard/pages/service-overview/components/tabs';
import * as _ from 'lodash';
import * as React from 'react';
import { useContext, useEffect, useState } from 'react';
import { useParams } from 'react-router-dom';
import {
  Button,
  ButtonType,
  Column,
  CoreLink,
  CoreText,
  Display,
  Grid,
  JustifyContent,
  StyledLayout,
  SVG,
  SVGAsset,
  TextType,
} from 'twitch-core-ui';

interface ServiceOverviewPageParams {
  id: string;
}

export const ServiceOverviewPage = () => {
  const { id } = useParams<ServiceOverviewPageParams>();
  const {
    service: fetchedService,
    loading,
    error,
    forbidden,
    notFound,
    reload,
  } = useService(id);

  const [service, setService] = useState(fetchedService);
  useEffect(() => setService(fetchedService), [fetchedService]);

  const modalContext = useContext(ModalContext);
  const handleShowEditModal = () =>
    modalContext.showModal({
      component: EditServiceModal,
      props: {
        service,
        onSaveSuccess: setService,
      },
    });

  if (forbidden) {
    return <Forbidden />;
  } else if (notFound) {
    return <NotFound />;
  } else if (loading) {
    return <StyledLoadingSpinner />;
  } else if (!_.isNil(error)) {
    return <Errored text={'Error fetching Service'} />;
  }

  return (
    <>
      <Grid>
        <Column cols={9}>
          <StyledLayout padding={1}>
            <CoreText className="service-overview-title" type={TextType.H3}>
              {service.name}
            </CoreText>{' '}
            <CoreLink
              className="service-overview-title"
              targetBlank
              linkTo={service.serviceCatalogUrl as string}
            >
              <SVG asset={SVGAsset.Popout} />
            </CoreLink>
            <CoreText type={TextType.H5}>LDAP: {service.ldapGroup}</CoreText>
          </StyledLayout>
        </Column>
        <Column cols={3}>
          <StyledLayout
            padding={1}
            display={Display.Flex}
            justifyContent={JustifyContent.End}
          >
            <Button
              type={ButtonType.Secondary}
              icon={SVGAsset.Edit}
              onClick={handleShowEditModal}
            >
              Edit service
            </Button>
          </StyledLayout>
        </Column>
        <Column cols={12}>
          <StyledLayout padding={1}>
            <ServiceOverviewTabs service={service} onReloadService={reload} />
          </StyledLayout>
        </Column>
      </Grid>
    </>
  );
};
