import { Errored } from 'dashboard/core/components/errored';
import { StyledLoadingSpinner } from 'dashboard/core/components/loading-spinner';
import { ModalContext } from 'dashboard/core/contexts/modals';
import { useServiceTargets } from 'dashboard/core/utils/target';
import { code } from 'dashboard/generated/controlplane';
import { RegisterTargetModal } from 'dashboard/pages/service-overview/components/register-target-modal';
import { TargetsTable } from 'dashboard/pages/service-overview/components/targets-table';
import * as _ from 'lodash';
import * as React from 'react';
import { useContext, useEffect, useState } from 'react';
import {
  AlignItems,
  Button,
  ButtonSize,
  CoreLink,
  CoreText,
  Display,
  FlexDirection,
  JustifyContent,
  Layout,
  Position,
  SVGAsset,
  TextType,
} from 'twitch-core-ui';

import twirp = code.justin.tv.eventbus.controlplane;

interface Props {
  service: twirp.IService;
}
export const SubscriptionTargets = ({ service }: Props) => {
  const { targets: fetchedTargets, loading, error } = useServiceTargets(
    service.id as string,
  );

  const { showModal } = useContext(ModalContext);

  const [targets, setTargets] = useState(fetchedTargets);
  useEffect(() => setTargets(fetchedTargets), [fetchedTargets]);

  const onDeregister = (target: twirp.ITarget) =>
    setTargets(_.filter(targets, (t) => t.id !== target.id));
  const onRegister = (target: twirp.ITarget) =>
    setTargets([...targets, target]);

  const showRegisterTargetModal = () =>
    showModal({
      component: RegisterTargetModal,
      props: {
        service,
        onSuccess: onRegister,
      },
    });

  return (
    <div>
      <Layout
        display={Display.Flex}
        justifyContent={JustifyContent.Between}
        alignItems={AlignItems.Center}
      >
        <Layout>
          <CoreText type={TextType.Span}>
            A subscription target represents the destination where you want
            events to be sent. For more information about subscription targets,
            visit the{' '}
            <CoreLink linkTo="https://git.xarth.tv/pages/eventbus/docs/subscribers/subscription_targets/">
              documentation
            </CoreLink>
            .
          </CoreText>
        </Layout>
        <Layout margin={{ left: 1 }}>
          <Button
            size={ButtonSize.Default}
            onClick={showRegisterTargetModal}
            icon={SVGAsset.AddReaction}
          >
            Register New Target
          </Button>
        </Layout>
      </Layout>
      <Layout
        display={Display.Flex}
        flexDirection={FlexDirection.Column}
        position={Position.Relative}
      />
      {loading && <StyledLoadingSpinner />}
      {error && <Errored text={'Error fetching targets...'} />}
      {!loading && !error && (
        <div>
          <TargetsTable
            service={service}
            targets={targets}
            onDeregister={onDeregister}
          />
        </div>
      )}
    </div>
  );
};
