import { ModalContext } from 'dashboard/core/contexts/modals';
import { code } from 'dashboard/generated/controlplane';
import { DeregisterTargetModal } from 'dashboard/pages/service-overview/components/deregister-target-modal';
import * as _ from 'lodash';
import * as React from 'react';
import { useContext } from 'react';
import {
  AlignItems,
  Background,
  Button,
  ButtonSize,
  ButtonType,
  Color,
  CoreLink,
  CoreText,
  Display,
  JustifyContent,
  Layout,
  StyledLayout,
  SVGAsset,
  Table,
  TableBody,
  TableCell,
  TableHeader,
  TableHeading,
  TableRow,
  TextAlign,
  TextType,
  VerticalAlign,
} from 'twitch-core-ui';
import { SubscriptionTargetValidation } from './validation';

import twirp = code.justin.tv.eventbus.controlplane;

interface Props {
  service: twirp.IService;
  targets: twirp.ITarget[];
  onDeregister: (target: twirp.ITarget) => void;
}

export const TargetsTable = ({ targets, onDeregister }: Props) => {
  const { showModal } = useContext(ModalContext);

  const showTargetDeregisterModal = (target: twirp.ITarget) => () =>
    showModal({
      component: DeregisterTargetModal,
      props: {
        target,
        onSuccess: onDeregister,
      },
    });

  const targetsByName = _.sortBy(targets, (target) =>
    _.lowerCase(target.name as string),
  );

  const rows: JSX.Element[] = _(targetsByName)
    .reject(_.isNil)
    .map((target) => {
      const queueUrl = _.get(target, 'sqs.queueUrl', 'Not found') as string;

      return (
        <TableRow key={target.id}>
          <TableCell verticalAlign={VerticalAlign.Middle}>
            <CoreText>{target.name}</CoreText>
          </TableCell>
          <TableCell verticalAlign={VerticalAlign.Middle}>{queueUrl}</TableCell>
          <TableCell
            verticalAlign={VerticalAlign.Middle}
            textAlign={TextAlign.Center}
          >
            <SubscriptionTargetValidation queueUrl={queueUrl} />
          </TableCell>
          <TableCell
            verticalAlign={VerticalAlign.Middle}
            textAlign={TextAlign.Center}
          >
            <CoreLink
              linkTo={`/services/${target.serviceId}/subscriptions/${target.id}`}
            >
              <Button
                type={ButtonType.Secondary}
                size={ButtonSize.Small}
                icon={SVGAsset.Gear}
              >
                {`Edit Subscriptions`}
              </Button>
            </CoreLink>
          </TableCell>
          <TableCell
            verticalAlign={VerticalAlign.Middle}
            textAlign={TextAlign.Center}
          >
            <Button
              type={ButtonType.Secondary}
              size={ButtonSize.Small}
              icon={SVGAsset.Close}
              onClick={showTargetDeregisterModal(target)}
            />
          </TableCell>
        </TableRow>
      );
    })
    .value();

  if (_.isEmpty(rows)) {
    return (
      <StyledLayout
        padding={1}
        display={Display.Flex}
        justifyContent={JustifyContent.Center}
      >
        <Layout padding={1} alignItems={AlignItems.Center}>
          <CoreText type={TextType.H4} color={Color.Alt2}>
            No targets found.
          </CoreText>
        </Layout>
      </StyledLayout>
    );
  }

  return (
    <Layout padding={{ top: 3 }}>
      <StyledLayout background={Background.Base} fullWidth>
        <Table>
          <TableHeader>
            <TableHeading label="Target Name" textAlign={TextAlign.Center} />
            <TableHeading label="SQS Queue URL" textAlign={TextAlign.Center} />
            <TableHeading label="Status" textAlign={TextAlign.Center} />
            <TableHeading label="Subscriptions" textAlign={TextAlign.Center} />
            <TableHeading label="" textAlign={TextAlign.Center} />
          </TableHeader>
          <TableBody>{rows}</TableBody>
        </Table>
      </StyledLayout>
    </Layout>
  );
};
