import { ModalContext } from 'dashboard/core/contexts/modals';
import {
  DEVELOPMENT,
  PRODUCTION,
  STAGING,
} from 'dashboard/definitions/environments';
import { code } from 'dashboard/generated/controlplane';
import { EventDetailsModal } from 'dashboard/pages/subscriptions/components/event-details-modal';
import * as _ from 'lodash';
import * as React from 'react';
import { useState } from 'react';
import {
  Button,
  ButtonSize,
  ButtonType,
  Column,
  Display,
  Grid,
  JustifyContent,
  Layout,
  StyledLayout,
  SVGAsset,
} from 'twitch-core-ui';

import twirp = code.justin.tv.eventbus.controlplane;

interface Props {
  eventStream: twirp.IEventStream;
  productionEnabled: boolean;
  stagingEnabled: boolean;
  developmentEnabled: boolean;
  onSubscribe: (env: string) => () => void;
  onUnsubscribe: (env: string) => () => void;
}
export const SubscriptionRow = ({
  eventStream,
  productionEnabled,
  stagingEnabled,
  developmentEnabled,
  onSubscribe,
  onUnsubscribe,
}: Props) => {
  const { showModal } = React.useContext(ModalContext);
  const handleShowEventDetailsModal = () => {
    showModal({
      component: EventDetailsModal,
      props: {
        eventStream,
      },
    });
  };

  const [hoverTarget, setHoverTarget] = useState('');
  const handleHover = (target: string) => {
    return () => {
      setHoverTarget(target);
    };
  };
  const handleUnhover = () => setHoverTarget('');

  return (
    <StyledLayout padding={{ y: 2 }} borderBottom>
      <Grid>
        <Column cols={6}>
          <Button type={ButtonType.Text} onClick={handleShowEventDetailsModal}>
            {eventStream.eventType}
          </Button>
        </Column>
        <Column cols={6}>
          <Layout
            display={Display.Flex}
            justifyContent={JustifyContent.Between}
            padding={{ x: 2 }}
          >
            <Button
              icon={buttonIcon(productionEnabled, hoverTarget === PRODUCTION)}
              size={ButtonSize.Small}
              type={buttonType(productionEnabled, hoverTarget === PRODUCTION)}
              onMouseEnter={handleHover(PRODUCTION)}
              onMouseLeave={handleUnhover}
              onClick={
                productionEnabled
                  ? onUnsubscribe(PRODUCTION)
                  : onSubscribe(PRODUCTION)
              }
            >
              {_.capitalize(PRODUCTION)}
            </Button>
            <Button
              icon={buttonIcon(stagingEnabled, hoverTarget === STAGING)}
              size={ButtonSize.Small}
              type={buttonType(stagingEnabled, hoverTarget === STAGING)}
              onMouseEnter={handleHover(STAGING)}
              onMouseLeave={handleUnhover}
              onClick={
                stagingEnabled ? onUnsubscribe(STAGING) : onSubscribe(STAGING)
              }
            >
              {_.capitalize(STAGING)}
            </Button>
            <Button
              icon={buttonIcon(developmentEnabled, hoverTarget === DEVELOPMENT)}
              size={ButtonSize.Small}
              type={buttonType(developmentEnabled, hoverTarget === DEVELOPMENT)}
              onMouseEnter={handleHover(DEVELOPMENT)}
              onMouseLeave={handleUnhover}
              onClick={
                developmentEnabled
                  ? onUnsubscribe(DEVELOPMENT)
                  : onSubscribe(DEVELOPMENT)
              }
            >
              {_.capitalize(DEVELOPMENT)}
            </Button>
          </Layout>
        </Column>
      </Grid>
    </StyledLayout>
  );
};

const buttonType = (subscribed: boolean, hovered: boolean): ButtonType => {
  if (subscribed && hovered) {
    return ButtonType.Alert;
  } else if (subscribed && !hovered) {
    return ButtonType.Primary;
  } else if (!subscribed && hovered) {
    return ButtonType.Success;
  } else {
    return ButtonType.Secondary;
  }
};

const buttonIcon = (subscribed: boolean, hovered: boolean): SVGAsset => {
  if (subscribed && hovered) {
    return SVGAsset.Trash;
  } else if (subscribed && !hovered) {
    return SVGAsset.Check;
  } else {
    return SVGAsset.Plus;
  }
};
