import { code } from 'dashboard/generated/controlplane';
import * as _ from 'lodash';
import * as React from 'react';
import * as Renderer from 'react-test-renderer';
import { Button, ButtonType, SVGAsset } from 'twitch-core-ui';
import { SubscriptionRow } from '.';

import twirp = code.justin.tv.eventbus.controlplane;

jest.mock('react-dom', () => ({
  // tslint:disable-next-line:no-any
  createPortal: (node: any) => node,
}));

const eventStream: twirp.IEventStream = {
  eventType: 'CoolEvent',
  environment: 'doesntactuallymatter',
};

describe('SubscriptionRow', () => {
  const root = Renderer.create(
    <SubscriptionRow
      eventStream={eventStream}
      // tslint:disable-next-line
      onSubscribe={(__: string) => _.noop}
      // tslint:disable-next-line
      onUnsubscribe={(__: string) => _.noop}
      productionEnabled={true}
      stagingEnabled={false}
      developmentEnabled={false}
    />,
  ).root;

  const prodButton = root.find(
    (node: Renderer.ReactTestInstance) =>
      node.type === Button && node.props.children === 'Production',
  );

  const stagingButton = root.find(
    (node: Renderer.ReactTestInstance) =>
      node.type === Button && node.props.children === 'Staging',
  );

  const devButton = root.find(
    (node: Renderer.ReactTestInstance) =>
      node.type === Button && node.props.children === 'Development',
  );

  it('has a prod button that is enabled', () => {
    expect(prodButton.props.icon).toBe(SVGAsset.Check);
  });

  it('has disabled staging and development buttons', () => {
    expect(stagingButton.props.icon).toBe(SVGAsset.Plus);
    expect(devButton.props.icon).toBe(SVGAsset.Plus);
  });

  describe('button hover behavior', () => {
    it('prod button before hover is primary color', () => {
      expect(prodButton.props.type).toBe(ButtonType.Primary);
    });
    it('prod button becomes a delete on hover', () => {
      Renderer.act(() => prodButton.props.onMouseEnter({}));
      expect(prodButton.props.type).toBe(ButtonType.Alert);
    });
    it('returns to primary after hover ends', () => {
      Renderer.act(() => prodButton.props.onMouseLeave({}));
      expect(prodButton.props.type).toBe(ButtonType.Primary);
    });
  });

  it('renders', () => {
    expect(
      Renderer.create(
        <React.Fragment>
          <SubscriptionRow
            eventStream={eventStream}
            // tslint:disable-next-line
            onSubscribe={(__: string) => _.noop}
            // tslint:disable-next-line
            onUnsubscribe={(__: string) => _.noop}
            productionEnabled={true}
            stagingEnabled={false}
            developmentEnabled={false}
          />
        </React.Fragment>,
      ),
    ).toMatchSnapshot();
  });
});
