package secretconf

import (
	"errors"
	"fmt"
	"reflect"

	"code.justin.tv/systems/sandstorm/manager"
)

type secretGetter interface {
	Get(secretName string) (*manager.Secret, error)
}

var _ secretGetter = (*manager.Manager)(nil)

const (
	secretKeyTag = "secret"
)

var byteSliceType = reflect.TypeOf([]byte(nil))
var stringType = reflect.TypeOf("") // nolint

// Load fills in a struct with sandstorm secrets secrets by inspecting the `secret` tag. Nested structs
// are recursively filled in. Pointers to structs are only filled in if non-nil.
//
// The `conf` parameter must be a pointer to a struct.
//
// The secret fetched from sandstorm is constructed from the team, service, environment values, and
// the secret tag i.e. <team>/<service>/<environment>/<secret tag> (format of all sandstorm secrets)
//
// Example struct:
//
// type Secrets struct {
//   APIKey string `secret:"api_key"`
//   APIKey2 []byte `secret:"api_key2"`
// }
//
// Example usage
//
//  var conf Secrets
//  role := "arn:aws:iam::111111111:role/sandstorm/production/templated/role/app-development"
//  manager := secretconf.NewManager(role)
//  err := secretconf.Load(&conf, manager, "foo", "bar", "development")
//  if err != nil {
//    log.Fatal(err)
//  }
//  log.Println(conf.APIKey)
//  log.Println(conf.APIKey2)
//
func Load(conf interface{}, sg secretGetter, team, service, environment string) error {
	if team == "" {
		return errors.New("team cannot be blank")
	} else if service == "" {
		return errors.New("service cannot be blank")
	} else if environment == "" {
		return errors.New("environment cannot be blank")
	}

	val := reflect.ValueOf(conf)
	// the conf must be a pointer for setting fields to make possible
	err := validatePointer(val)
	if err != nil {
		return err
	}

	// Elem() deferences the pointer
	return loadStruct(val.Elem(), sg, team, service, environment)
}

func loadStruct(elem reflect.Value, sg secretGetter, team, service, environment string) error {
	// only structs make sense
	err := validateStruct(elem)
	if err != nil {
		return err
	}

	confType := reflect.TypeOf(elem.Interface())

	// iterate over each field. if it has a `secret` tag then fetch
	// the secret and set the secret to the field
	for i := 0; i < elem.NumField(); i++ {
		elemField := elem.Field(i)

		// skip invalid fields
		if !elemField.IsValid() {
			continue
		}

		fieldType := confType.Field(i)

		kind := elemField.Kind()

		var structToLoad reflect.Value

		// Recursively set structs or pointer to structs
		if kind == reflect.Struct {
			structToLoad = elemField
		} else if kind == reflect.Ptr && !elemField.IsNil() {
			f := elemField.Elem()
			if f.Kind() == reflect.Struct {
				structToLoad = f
			}
		}

		if structToLoad.IsValid() {
			loadErr := loadStruct(structToLoad, sg, team, service, environment)
			if loadErr != nil {
				return fmt.Errorf("error loading struct field %s: %v", fieldType.Name, loadErr)
			}
			continue
		}

		// skip unsettable fields
		if !elemField.CanSet() {
			continue
		}

		// make sure the field type is a string or []byte
		isSupportedType := kind == reflect.String ||
			(kind == reflect.Slice && elemField.Type() == byteSliceType)

		if !isSupportedType {
			continue
		}

		// skip fields without a secret tag or blank value
		secretKey := fieldType.Tag.Get(secretKeyTag)
		if secretKey == "" {
			continue
		}

		// fetch the secret
		var secret *manager.Secret
		secretPath := formatSecret(team, service, environment, secretKey)
		secret, err = sg.Get(secretPath)
		if err != nil {
			return fmt.Errorf("failed to get secret %q: %v", secretPath, err)
		}

		if secret == nil {
			return fmt.Errorf("received nil secret; secret %q may not exist in sandstorm", secretPath)
		}

		// set the secret
		switch elemField.Kind() {
		case reflect.String:
			elemField.SetString(string(secret.Plaintext))
		case reflect.Slice:
			elemField.Set(reflect.ValueOf(secret.Plaintext))
		}
	}

	return nil
}

func formatSecret(team, service, environment, name string) string {
	return fmt.Sprintf("%s/%s/%s/%s", team, service, environment, name)
}

func validatePointer(val reflect.Value) error {
	if val.Kind() != reflect.Ptr || val.IsNil() {
		return errors.New("conf must be a pointer and not nil")
	}

	return nil
}

func validateStruct(val reflect.Value) error {
	if val.Kind() != reflect.Struct {
		return errors.New("val must be a struct pointer")
	}

	return nil
}
