const path = require("path");
const webpack = require("webpack");
const CleanWebpackPlugin = require("clean-webpack-plugin");
const HtmlWebpackPlugin = require("html-webpack-plugin");
const MiniCssExtractPlugin = require("mini-css-extract-plugin");

module.exports = (env, argv) => {
  const isProd = argv.mode === "prod";

  const config = {
    entry: "./src/index.tsx",

    output: {
      filename: isProd ? "[name].[contenthash].js" : "[name].js",
      path: path.join(__dirname, "/dist"),
      publicPath: "/",
    },

    devServer: {
      contentBase: "public",
      proxy: {
        "/api": "http://localhost:8000",
      },
      historyApiFallback: true,
    },

    optimization: {
      runtimeChunk: "single",
      splitChunks: {
        cacheGroups: {
          vendor: {
            test: /[\\/]node_modules[\\/]/,
            name: "vendors",
            chunks: "all",
          },
        },
      },
    },

    resolve: {
      // Add reference so we can absolutely refer to dashboard paths
      alias: {
        dashboard: path.join(__dirname, "src"),
        "twitch-core-ui": path.join(
          __dirname,
          "node_modules/twitch-core-ui/module"
        ),
      },
      // Add '.ts' and '.tsx' as resolvable extensions.
      extensions: [".ts", ".tsx", ".js", ".json"],
    },

    plugins: [
      new CleanWebpackPlugin(),
      new MiniCssExtractPlugin({
        filename: isProd ? "[name].[contenthash].css" : "[name].css",
      }),
      new webpack.HashedModuleIdsPlugin(),
      new HtmlWebpackPlugin({
        template: path.join(__dirname, "/public/index.html"),
        filename: path.join(__dirname, "/dist/index.html"),
      }),
    ],

    module: {
      rules: [
        {
          test: /\.(ico|png|jpg|gif|svg|eot|ttf|woff|woff2|swf)$/,
          loader: "file-loader",
        },
        // All files with a '.ts' or '.tsx' extension will be handled by 'ts-loader'.
        { test: /\.tsx?$/, loader: "ts-loader" },

        // All output '.js' files will have any sourcemaps re-processed by 'babel-loader'.
        // NOTE: might need to exclude node_modules here if running into issues
        {
          enforce: "pre",
          test: /\.js$/,
          exclude: /node_modules/,
          loader: "babel-loader",
        },

        {
          test: /\.s?[ac]ss$/,
          use: [
            isProd ? MiniCssExtractPlugin.loader : "style-loader",
            "css-loader",
            {
              loader: "postcss-loader",
              options: {
                plugins: [
                  require("autoprefixer")(),
                  require("css-mqpacker")({ sort: true }),
                ],
              },
            },
            {
              loader: "sass-loader",
              options: {
                includePaths: ["node_modules/twitch-core-ui/src/scss"],
              },
            },
          ],
        },
      ],
    },
  };

  if (!isProd) {
    // Enable sourcemaps for debugging webpack's output.
    config.devtool = "source-map";
    config.plugins.push(new webpack.HotModuleReplacementPlugin());
  }

  return config;
};
