---
title: Getting Started
weight: 2
---

{{% notice note %}}
You must be added to the `infra` LDAP group in order to access the eventbus dashboard. If you are not in `infra`, file a ticket with IT [here](https://jira.twitch.com/servicedesk/customer/portal/1/create/18).
{{% /notice %}}

There are two steps necessary to onboard onto the Event Bus. After completing these steps your service will be able to publish and subscribe to events.

# Step 1: Add your Service to the Dashboard (5 minutes)

Services are registered in the [Event Bus dashboard](https://eventbus.xarth.tv). Be sure that your service exists in the [Twitch Service Catalog](https://catalog.xarth.tv/services) prior to registration.

From the dashboard, click the **Add Your Service** button. Select your LDAP Group from the dropdown menu and input your service's Service Catalog entry. After filling out the necessary fields, click **Register Service**. Then, add the [IAM Roles]({{< ref "iam_roles/_index.md" >}}) for all your service's environments, which will be used later to permission access to resources.

![Add Service](../images/register_new_service.gif)

{{% notice note %}}
The LDAP group you select will have the ability to modify the service and its subscriptions. Be sure to select the LDAP group for the team that owns the service.
{{% /notice %}}

# Step 2: Integrate the CloudFormation Template (15 minutes)

The Event Bus automatically spins up and wires together infrastructure for you. To do this properly, you must provide the Event Bus the necessary permissions to create and modify a small subset of infrastructure.

Additionally, you must provide your service the permissions necessary to contact the Event Bus in order to publish and/or subscribe. As the Event Bus evolves to support new features, the required permissions may change in the future.

To simplify the process, we have created a CloudFormation template for you. This template requests the minimum permissions necessary and has very limited scope. We recommend examining the CloudFormation template for yourself [here](https://eventbus-setup.s3-us-west-2.amazonaws.com/cloudformation.yaml).

## Part 1: Run the CloudFormation Template

{{% notice note %}}
If your AWS account already has a CloudFormation stack with the name "EventBus", this step has already been completed.
{{% /notice %}}

Examples in this documentation use the latest version of the CloudFormation template, however if you wish to pin your version consult [CloudFormation Versions]({{< ref "getting_started/cloudformation_versions.md" >}}).

{{% expand_markdown "From non-Fulton (Terraform)" %}}

```tf
data "http" "eventbus_cloudformation" {
  url = "https://eventbus-setup.s3-us-west-2.amazonaws.com/cloudformation.yaml"
}

resource "aws_cloudformation_stack" "eventbus" {
  name          = "EventBus"
  capabilities  = ["CAPABILITY_NAMED_IAM"]
  template_body = "${data.http.eventbus_cloudformation.body}"
}
```

{{% /expand_markdown %}}

{{% expand_markdown "From non-Fulton (CloudFormation)" %}}

```yaml
EventBus:
  Type: AWS::CloudFormation::Stack
  Properties:
    TemplateURL: "https://eventbus-setup.s3-us-west-2.amazonaws.com/cloudformation.yaml"
```

{{% /expand_markdown %}}

{{% expand_markdown "From Fulton (CloudFormation)" %}}

The following is only supported by the LPT based Fulton setup, the Fulton Construct for CDK is coming soon. Meanwhile, if you are using Fulton CDK, you can spin up the EventBus stack manually or make an intermediate wrapper similar to [this example](https://code.amazon.com/reviews/CR-43041977/revisions/2#/diff). This involves [wrapping the EventBus CloudFormation template](https://code.amazon.com/packages/RareCandyCDK/blobs/mainline/--/lib/stacks/eventbus/index.ts) and [initializing the setup](https://code.amazon.com/packages/RareCandyCDK/blobs/49b3e120874e0d052c9e4afa05f235b1b0edf2ec/--/lib/pipeline-stages.ts#L174-L180). We recommend manually creating the stack until we provide the Fulton Construct but this intermediate solution is possible if necessary.

```yaml
  Twitch-Eventbus-Setup = 1.0;
```

1. Navigate to your package's `VpcInfrastructure` [note: this is NOT your main package, search [YourPackageName]VPCInfrastructure on code.amazon.com].
1. Navigate to the [YourPackageName]VPCInfrastructure's Brazil `Config` file. It should be in the root directory.
1. Add `Twitch-Eventbus-Setup` to the list of `dependencies` in the `Config` file.
  - Follow this [guide](https://docs.fulton.twitch.a2z.com/docs/adding_dependencies.html#adding-a-go-dependency-to-your-project) in the Fulton docs to add the dependency.
1. See an example [here](https://code.amazon.com/packages/HomeroomVpcInfrastructure/blobs/mainline/--/Config#L32).

{{% /expand_markdown %}}

{{% expand_markdown "From Fulton (CDK)" %}}

{{% notice note %}}
The EventBus integration for Fulton CDK is new! Help make it better by raising any issues encountered in #event-bus.
{{% /notice %}}

Make the following changes to your services `CDK` package (e.g. `MyFoobarServiceCDK`)

1. Add the following import to `Config`: `TwitchEventBusIntegration = 1.x`
2. In `package.json` include:
```
{
  "dependencies": {
    "@twitch/eventbus-integration": "*"
  }
}
```
3. Add the construct to your **VPC stack** (file should be `lib/stacks/vpc-stack.ts`):

```
import { DeploymentStack, DeploymentStackProps } from '@amzn/pipelines';
import * as ec2 from 'monocdk/aws-ec2';
import * as cdk from 'monocdk';
import * as fulton from '@twitch/fulton';

// vvv Add this line
import * as eventbusIntegration from '@twitch/eventbus-integration';
// ^^^

// ...

export class VPCStack extends DeploymentStack {
  readonly vpc: ec2.IVpc;

  constructor(parent: cdk.Construct, name: string, props: VPCStackProps) {
    super(parent, name, props);

    // ... other VPC infrastructure

    // vvv Add this line
    new eventbusIntegration.EventBusIntegration(this, 'EventBusIntegration');
    // ^^^

    // ... other VPC infrastructure
  }
}
```


{{% /expand_markdown %}}

{{% expand_markdown "From the AWS Console" %}}

1. Go into the AWS console and click **Services > CloudFormation**
1. Click the **Create stack** button
1. Select "Template is ready"
1. Select "Amazon S3 URL"
1. Enter `https://eventbus-setup.s3-us-west-2.amazonaws.com/cloudformation.yaml`
1. Click **Next**
1. Name the stack "EventBus" then click **Next** twice
1. Acknowledge the pop-up and click **Create**

{{% /expand_markdown %}}

## Part 2: Attach the Event Bus Policy {#attach-event-bus-policy}

Your service will need permission to interact with the Event Bus in order to publish, subscribe, encrypt/decrypt payloads, etc. To do this, attach the "EventBusAccess" policy to your application's IAM role.

{{% expand_markdown "From non-Fulton (Terraform)" %}}

```tf
resource "aws_iam_role_policy_attachment" "eventbus_access_attach" {
  role       = "${aws_iam_role.role_your_application_uses.name}"
  policy_arn = "${aws_cloudformation_stack.eventbus.outputs["EventBusAccessPolicyARN"]}"
}
```

{{% /expand_markdown %}}

{{% expand_markdown "From non-Fulton (CloudFormation)" %}}

Make sure you have an output for the access policy defined.

```yaml
Outputs:
  EventBusAccessPolicyARN:
    Description: "EventBusAccess policy ARN"
    Value: ...
    Export:
      Name: "eventbus-access-policy-arn"
```

Wherever you create the `AWS::IAM::Role` for your application, in the `ManagedPolicyArns` section add:

```yaml
ManagedPolicyArns:
  - !GetAtt EventBus.Outputs.EventBusAccessPolicyARN
```

{{% /expand_markdown %}}

{{% expand_markdown "From Fulton (CloudFormation)" %}}

If the EventBus resources were created with the Fulton step from the previous section, they will be embedded in the VPC infrastructure stack. In this case, use the global reference since there will be no local EventBus stack.

Verify that you have indeed created the EventBus resources. You can do this by accessing the VPC stack from within the CloudFormation section of your AWS account. Navigate to the Resources tab and enter "eventbus" in the search box. You should see 4 resources (`EventBusAccessPolicy`, `EventBusPublisherRoleV1`, `EventBusRole`, and `EventBusSubscriberRoleV1`). Next, navigate to the Outputs tab and search "eventbus" again. Now, you should see 3 outputs (`EventBusAccessPolicyARN`, `EventBusKMSMasterKeyARN`, `EventBusSQSPolicyDocument`).

If this is all present, you will then be able to do the following to reference the access policy from your service role:

```yaml
ManagedPolicyArns:
  -  Fn::ImportValue: "eventbus-access-policy-arn"
```
{{% /expand_markdown %}}

{{% expand_markdown "From Fulton (CDK)" %}}

{{% notice note %}}
The EventBus integration for Fulton CDK is new! Help make it better by raising any issues encountered in #event-bus.
{{% /notice %}}

In the service's `CDK` package, locate the runtime IAM role (for Lambda, it is named `lambdaRole` in file ) and add the following code to attach the `EventBusAccess` managed policy:

**Lambda**

In `lib/lambda-service.ts`, locate `lambdaRole` and add this:

```
lambdaRole.addManagedPolicy(
  iam.ManagedPolicy.fromManagedPolicyArn(
    this,
    'eventbusAccessPolicy',
    Fn.importValue('eventbus-access-policy-arn'),
  ),
);
```

**ECS on Fargate or EC2**

In `lib/service.ts`, locate `this.taskDefinition` and add this:

```
this.taskDefinition.taskRole.addManagedPolicy(
  iam.ManagedPolicy.fromManagedPolicyArn(
    this,
    'eventbusAccessPolicy',
    Fn.importValue('eventbus-access-policy-arn'),
  ),
);
```

{{% /expand_markdown %}}

{{% expand_markdown "From the AWS Console" %}}

1. Go into the AWS console and click **Services > IAM**
1. Click the **Policies** button on the left tab
1. Search for "EventBusAccess" and select it by clicking the circle to the left of its name
1. Click the **Policy actions** dropdown and select **Attach**
1. Search for your application's role and select it by clicking the checkbox to the left of its name
1. Click the **Attach policy** button

{{% /expand_markdown %}}
