---
title: Testing Locally
weight: 2
---

The Event Bus client library provides a couple of methods for local testing, depending on the "level" you want to work at.


## Mock Publisher

We provide a [mock Publisher][mock-pub] object that implements the `eventbus.Publisher` interface. This mock also keeps a synchronized list of the calls made so you can test how it got called. Of course, being a single-method interface, it's also easy to use your favorite mocking library (e.g. testify) if you prefer.

In your test function, create a mock publisher. For example:

```go
import mockEventbusClient "code.justin.tv/eventbus/client/mock"

mockEventbus := &mockEventbusClient.Publisher{}
```

Inject the mock publisher into your function, struct, etc:

```go
job := &Job{
	// ...
	eventbus: mockEventbus,
	// ...
}
```

Create a handler on publish to capture the calls being made in your application:

```go
import (
	eventbus "code.justin.tv/eventbus/client"
	mockEventbusClient "code.justin.tv/eventbus/client/mock"
)

mockEventbus.PublishHandler = func(pub *mockEventbusClient.Publisher, ctx context.Context, event eventbus.Message) error {
	userSubscribeUpdate := event.(*user_subscribe_user.Update) // check for you particular event here
	assert.Equal(t, "123", userSubscribeUpdate.ToUserId)       // make assertions based on your testing logic and the expected event data
	// ...
}

// check the mocked publisher's stored number of calls
assert.Equal(t, 1, len(mockEventbus.PublishCalls), "should have one call to eventbus publish")
```

Call the functionality that will trigger your mock:

```go
err := job.DoWork()
```

[mock-pub]: https://godoc.internal.justin.tv/code.justin.tv/eventbus/client/mock#Publisher

## Publishing in Local Mode

If you want a more "functional" or "end-to-end" style test (e.g. using `localstack` to create a local SNS topic), modify the AWS config:

```go
config := aws.NewConfig().
    WithRegion("us-east-1").
    WithEndpoint("http://containername:1234")

p, err := publisher.New(publisher.Config{
	EventTypes:   []string{...},
	Environment:  publisher.EnvLocal,
	Session:      session.Must(session.NewSession(config)),
})
```

If you want to customize the SNS topic names, you will want to provide a custom `RouteFetcher`.

## Test Payloads

Sometimes it's useful to create test payloads, see our [subscriber testing docs]({{< ref "/subscribers/testing.md#creating-test-payloads" >}}) for a detailed overview of how to do this.
