---
title: Subscribers
weight: 3
---

This section assumes you have followed the steps in [Getting Started][].

[Getting Started]: {{< ref "getting_started/_index.md" >}}

## Create a Subscription Target {#create-target}

A subscription target allows you to subscribe to one or more event streams. You can create as many subscription targets as your service needs. For more information about subscription targets, visit the [Subscription Targets]({{< ref "subscribers/subscription_targets.md" >}}) page.

First, create the necessary infrastructure for your subscription target. For example, if you're creating an SQS queue, follow the instructions in the [SQS Queues]({{< ref "subscribers/sqs_queues.md" >}}) page.

Next, find your service in the [My Services](https://eventbus.xarth.tv/services) page of the Event Bus dashboard. Click your service's name to be taken to the service overview page. Next, within the **Subscription Targets** tab, click **Register New Target**. Provide your SQS queue URL as well as a human friendly name.

![Add Subscription Target](../images/register_subscription_target.gif)


## Subscribe to Events {#subscribe}

Find the subscription target you created in the previous section, then click the **Edit Subscriptions** button. The Subscriptions page will load.

Event types that the subscription target is currently subscribed to will appear at the top of the page. To enable or disable subscriptions, find the event type you are interested in and click the environment you are interested in subscribing to, or unsubscribing from, the subscription target.

![Add Subscriptions](../images/add_subscriptions.gif)

{{% notice note %}}
Before changing your subscriptions, consider updating your application code first to handle the new event types. Otherwise events will start queueing until your application code has been updated to process those events.
{{% /notice %}}

## Handle Events in your Application

Create an Event Bus mux to handle all the events you will receive. In this example, we listen for the `ClockUpdate` event:

```go
import (
    "context"
    "fmt"
    "log"

    eventbus "code.justin.tv/eventbus/client"
    "code.justin.tv/eventbus/schema/pkg/clock"
)

func clockHandler(ctx context.Context, h *eventbus.Header, event *clock.Update) error {
    log.Infof("Got clock tick: %d", event.Time.Seconds)
    return nil
}

func main() {
    mux := eventbus.NewMux()
    clock.RegisterUpdateHandler(mux, clockHandler)
}
```

Note that you registered your handler with a mux, which works similarly to those used in HTTP libraries. You can register multiple handlers with the same mux, which will dispatch based on the event type received. Returning a `nil` error from your handler causes the message to be marked successful. Returning any non-`nil` error will cause the event to be retried (most likely on another machine/worker) if your subscription target supports retries.

Next up, decide which client you would like to use: [SQS Client]({{< ref "#sqs-client" >}}) or [Lambda Client]({{< ref "#lambda-client" >}}).

### SQS Client

If you're considering processing messages directly from the SQS queue, be sure to read the [SQS Queues]({{< ref "subscribers/sqs_queues.md" >}}) page.

Create a new Event Bus subscriber client, using the `mux` you created earlier. Be sure to specify the correct SQS queue URL, which you'll find on the Subscription Targets page of the dashboard.

```go
import (
    "github.com/aws/aws-sdk-go/aws/session"
    "code.justin.tv/eventbus/client/subscriber/sqsclient"
)

func main() {
    // set up mux as before

    awsSession, err := session.NewSession()

    client, err := sqsclient.New(sqsclient.Config{
        Session: awsSession,
        Dispatcher: mux.Dispatcher(),
        QueueURL: "<sqs queue url from event bus dashboard>",
    })
}
```

Note that calling `mux.Dispatcher()` returns an immutable dispatcher object, so add all your handlers *before* calling `mux.Dispatcher()`.

If your service supports graceful shutdown, be sure to gracefully shut down your Event Bus subscriber client too!

```go
client.Shutdown()
```

For advanced configuration of the client, visit the [Client Configuration]({{< ref "subscribers/client_configuration.md" >}}) page.

### Lambda Handler

If you're considering using a Lambda to process messages, be sure to read the [Lambda Handlers]({{< ref "subscribers/lambda_handlers.md" >}}) page.

Simply start a new Event Bus Lambda handler, using the `mux` you created earlier:

```go
import (
    "github.com/aws/aws-lambda-go/lambda"
    "code.justin.tv/eventbus/client/subscriber/lambdafunc"
)

func main() {
    // set up mux as before

    lambda.Start(lambdafunc.NewSQS(mux.Dispatcher()))
}
```
