---
title: Authorized Fields
weight: 5
---

{{% authorized-field-warning %}}

# Step 0: Prerequisites

This documentation assumes that you have already onboarded your service to the EventBus via the dashboard, registered a subscription target, and added a subscription to an event stream from which you want to read an authorized field. For those tasks, see the [subscriber documentation]({{< ref "subscribers" >}}).

# Step 1: Generate an Authorized Field Access Request

Navigate to the [EventBus event catalog](https://eventbus.xarth.tv/events) and find the event and authorized field you want access to. Click the **Request Access** button.

You will be asked for the following information

* **Service:** The EventBus service to receive the authorized field access
* **IAM Role:** The IAM role, belonging to the service, that should be granted permissions to access the authorized field
* **Environment:** Production, staging, or development. 

Access to the authorized field will only be given to the chosen IAM role, and only for events from the chosen environment. Click **Generate**.

After clicking the **Generate** button, you will be given a link, as well as a list the people belonging to the LDAP group which owns the event type.

# Step 2: Reach Out to the Event Type's Owner LDAP Group

Before reaching out to any members of the LDAP group from Step 1, first ensure that your service meets the [Twitch data handling standards](https://wiki.xarth.tv/display/SEC/Twitch+Data+Handling+Standard) for the classification level of the authorized field you are requesting access to.

When ready, reach out to the event type owners and provide the generated link. They can paste it into their browser for a prefilled form granting access to your IAM role (NOTE: they should do this after logging into the dashboard, or else they will be redirected to the homepage after authenticating!).

{{% notice note %}}
You may need to follow the above instructions multiple times if you require access for several IAM roles (e.g. production IAM role and staging IAM role, etc)
{{% /notice %}}

# Step 3: Decrypt Authorized Fields in Your Application

Decrypting authorized fields as a subscriber requires use of a client that implements `authorization.Decrypter` within event dispatch handlers. The EventBus client provides an implementation in the `encryption` package, `AuthorizedFieldClient`. The following code shows a before and after, adding logic to access an authorized field called `email` on the `UserUpdate` event:

_Before_

```
import (
    // ...
)

func main() {
    mux := eventbus.NewMux()

    user.RegisterUpdateHandler(mux, userUpdateHandler)
}

func userUpdateHandler(ctx context.Context, h *eventbus.Header, u *user.Update) error {
    fmt.Printf("Received UserUpdate: %v\n", u)
    return nil
}
```

_After_

```
import (
    // ...

    "code.justin.tv/eventbus/client/encryption"
)

func main() {
    mux := eventbus.NewMux()

    sess, _ := session.NewSession() // AWS session
    authorizedFieldClient, err := encryption.NewAuthorizedFieldClient(sess)
    if err != nil {
        log.Fatal(err)
    }

    myHandlers := &SubscriptionHandler{
        decrypter: authorizedFieldClient,
    }

    user.RegisterUpdateHandler(mux, myHandlers.userUpdateHandler)
   
}

// SubscriptionHandler gives handler functions access to an authorized field decrypter
type Handler struct {
  decrypter encryption.Decrypter
}

func (h *Handler) userUpdateHandler(ctx context.Context, h *eventbus.Header, u *user.Update) error {
    fmt.Printf("Received UserUpdate: %v\n", u)

    // Get the decrypted value by calling a special 'GetDecrypted<FieldName>' method on the containing
    // protobuf message, and passing in the decrypter initialized for the handler
    email, err := u.GetDecryptedEmail(h.decrypter)
    if err != nil {
        return errors.Wrap(err, "could not access email authorized field")
    }

    fmt.Println("Decrypted email: " + email)

    return nil
}
```
