provider "aws" {
  version = "~> 2.0"
  region  = "us-west-2"
}

locals {
  region         = "us-west-2"
  environment    = "e2e-test"
  account_name   = "twitch-eventbus-main-test"
  account_id     = "793846415324"
  s3_bucket_name = "eventbus-config-e2e.internal.justin.tv"
}

terraform {
  required_version = ">= 0.11"

  backend "s3" {
    bucket = "twitch-eventbus-main-test"
    key    = "tfstate/eventbus/twitch-eventbus-main-test/terraform.state"
    region = "us-west-2"
  }
}

# Enable KMS for E2E testing
resource "aws_kms_key" "encryption_at_rest_key" {
  lifecycle {
    ignore_changes = ["policy"]
  }

  description = "SNS and SQS encryption at rest CMK"

  policy = <<EOF
{
    "Version": "2008-10-17",
    "Statement": [
        {
            "Sid": "Enable IAM User Permissions - EventBus",
            "Effect": "Allow",
            "Principal": {
                "AWS": "arn:aws:iam::${local.account_id}:root"
            },
            "Action": "kms:*",
            "Resource": "*"
        },
        {
            "Sid": "Enable IAM User Permissions - Publishers",
            "Effect": "Allow",
            "Principal": {
                "AWS": [
                    "arn:aws:iam::${local.account_id}:root"
                ]
            },
            "Action": [
                "kms:Decrypt",
                "kms:GenerateDataKey"
            ],
            "Resource": "*"
        },
        {
            "Sid": "Enable Service Permissions",
            "Effect": "Allow",
            "Principal": {
                "Service": [
                    "sqs.amazonaws.com",
                    "sns.amazonaws.com"
                ]
            },
            "Action": [
                "kms:Decrypt",
                "kms:GenerateDataKey"
            ],
            "Resource": "*"
        }
    ]
}
EOF
}

resource "aws_kms_alias" "encryption_at_rest_key" {
  name          = "alias/sns-sqs-encryption-key"
  target_key_id = "${aws_kms_key.encryption_at_rest_key.key_id}"
}

resource "aws_kms_key" "authorized_fields_key" {
  lifecycle {
    ignore_changes = ["policy"]
  }

  description = "Used to encrypt and decrypt authorized EventBus fields"

  policy = <<EOF
{
    "Version": "2008-10-17",
    "Statement": [
        {
            "Sid": "Enable IAM User Permissions - EventBus",
            "Effect": "Allow",
            "Principal": {
                "AWS": "arn:aws:iam::${local.account_id}:root"
            },
            "Action": "kms:*",
            "Resource": "*"
        }
    ]
}
EOF
}

resource "aws_kms_alias" "authorized_fields_key" {
  name          = "alias/field-level-encryption-key"
  target_key_id = "${aws_kms_key.authorized_fields_key.key_id}"
}

# TODO: allow jenkins to assume this role
resource "aws_iam_role" "e2e_test_runner" {
  name = "e2e-test-runner"

  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": "sts:AssumeRole",
      "Principal": {
        "AWS": [
          "arn:aws:iam::793846415324:role/Admin",
          "arn:aws:iam::043714768218:role/jenkins-prod-slave"
        ]
      },
      "Effect": "Allow"
    }
  ]
}
EOF
}

module "controlplane_policies" {
  source    = "../../modules/e2e_controlplane_policies"
  role_name = "${aws_iam_role.e2e_test_runner.name}"
}

# Allow the e2e test runner to assume role into the pub and sub accounts
resource "aws_iam_role_policy" "assume_role_policy" {
  name = "assume-role-policy"
  role = "${aws_iam_role.e2e_test_runner.id}"

  policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": {
    "Effect": "Allow",
    "Action": "sts:AssumeRole",
    "Resource": [
      "arn:aws:iam::567312298267:role/e2e-publisher-role",
      "arn:aws:iam::311539646041:role/e2e-subscriber-role"
    ]
  }
}
EOF
}

# Put CFN in S3 for pub and sub roles to use to enable E2E publishing and subscribing runs
module "cross_account_cloudformation" {
  source = "../../modules/cross_account_setup"

  environment                   = "${local.environment}"
  account_id                    = "${local.account_id}"
  s3_bucket                     = "${local.s3_bucket_name}"
  encryption_at_rest_cmk_key_id = "${aws_kms_key.encryption_at_rest_key.key_id}"
  authorized_fields_cmk_key_id  = "${aws_kms_key.authorized_fields_key.key_id}"
  latest_version                = "v1.4.0"
}
