provider "aws" {
  version = "~> 2.0"
  region  = "us-west-2"
}

locals {
  region                              = "us-west-2"
  environment                         = "prod"
  account_name                        = "twitch-eventbus-prod"
  account_id                          = "859517684765"
  s3_bucket_name                      = "eventbus-config.internal.justin.tv"
  slack_channel                       = "#event-bus-team"
  dashboard_app_count                 = 3
  dashboard_task_memory               = 1024
  dashboard_task_cpu                  = 512
  controlplane_httpserver_app_count   = 3
  controlplane_httpserver_task_memory = 1024
  controlplane_httpserver_task_cpu    = 512
  controlplane_converger_app_count    = 1
  controlplane_converger_task_memory  = 1024
  controlplane_converger_task_cpu     = 512
  instance_type                       = "m5.large"

  controlplane_validator_task_memory   = 1024
  controlplane_validator_task_cpu      = 512
  controlplane_validator_app_count     = 1
  controlplane_validator_slack_channel = "#eventbus-validation"

  pagerduty_high_urgency_key = "c02717fd9aac41558ec0637226e5751e"
  pagerduty_low_urgency_key  = "44469053715d46adadbbf949cdc9d444"
}

terraform {
  required_version = ">= 0.11"

  backend "s3" {
    bucket = "twitch-eventbus-prod"
    key    = "tfstate/eventbus/twitch-eventbus-prod/terraform.state"
    region = "us-west-2"
  }
}

data "terraform_remote_state" "account" {
  backend = "s3"

  config = {
    bucket = "twitch-eventbus-prod"
    key    = "tfstate/eventbus/twitch-eventbus-prod/account/terraform.tfstate"
    region = "us-west-2"
  }
}

module "app" {
  source = "../../modules/app"

  environment                         = local.environment
  region                              = local.region
  controlplane_httpserver_image       = "${local.account_id}.dkr.ecr.${local.region}.amazonaws.com/controlplane-httpserver:latest"
  controlplane_converger_image        = "${local.account_id}.dkr.ecr.${local.region}.amazonaws.com/controlplane-converger:latest"
  dashboard_image                     = "${local.account_id}.dkr.ecr.${local.region}.amazonaws.com/dashboard:latest"
  ec2_instance_type                   = local.instance_type
  cluster_size                        = var.cluster_size
  controlplane_httpserver_app_count   = local.controlplane_httpserver_app_count
  controlplane_httpserver_task_memory = local.controlplane_httpserver_task_memory
  controlplane_httpserver_task_cpu    = local.controlplane_httpserver_task_cpu
  controlplane_converger_app_count    = local.controlplane_converger_app_count
  controlplane_converger_task_memory  = local.controlplane_converger_task_memory
  controlplane_converger_task_cpu     = local.controlplane_converger_task_cpu
  dashboard_app_count                 = local.dashboard_app_count
  dashboard_task_memory               = local.dashboard_task_memory
  dashboard_task_cpu                  = local.dashboard_task_cpu
  rds_reader_hostname                 = module.rds.reader_hostname
  rds_writer_hostname                 = module.rds.writer_hostname
  slack_channel                       = local.slack_channel
  authorized_field_key_arn            = aws_kms_key.field_level_encryption_key.arn
  encryption_at_rest_key_arn          = aws_kms_key.encryption_at_rest_key.arn

  # override the non-production default dns servers
  # https://wiki.twitch.com/display/SYS/What+DNS+servers+should+be+used
  dns_servers = ["10.203.126.62", "10.203.124.218", "10.203.125.211", "10.203.126.205"]

  controlplane_validator_image                      = "${local.account_id}.dkr.ecr.${local.region}.amazonaws.com/controlplane-validator:latest"
  controlplane_validator_app_count                  = local.controlplane_validator_app_count
  controlplane_validator_task_memory                = local.controlplane_validator_task_memory
  controlplane_validator_task_cpu                   = local.controlplane_validator_task_cpu
  controlplane_validator_slack_channel              = local.controlplane_validator_slack_channel
  controlplane_validator_pagerduty_enabled          = true
  controlplane_validator_pagerduty_high_urgency_key = local.pagerduty_high_urgency_key
  controlplane_validator_pagerduty_low_urgency_key  = local.pagerduty_low_urgency_key

}

module "osiris_stack_eventbus_dashboard" {
  source                = "../../modules/osiris_stack"
  osiris_service_name   = "eventbus-dashboard-production"
  team_name             = "Edge Infrastructure"
  team_email            = "eventbus@twitch.tv"
  additional_domains    = ["eventbus.internal.justin.tv", "eventbus.xarth.tv"]
  internal_fqdn         = "dashboard.prod.eventbus.twitch.a2z.com"
  osiris_pf_integration = "EventbusDashboardProd-osiris_IDP"
  sandstorm_role        = "osiris-eventbus-dashboard-production"
  sandstorm_prefix      = "eventbus/dashboard/production"
  vpc_id                = "vpc-0977d2d3037202c79"
  public_subnets        = ["subnet-07ec5000ed6b3563e", "subnet-0e5aae88722d67a83", "subnet-0c21e40be39f145bf"]
  private_subnets       = ["subnet-00a5db39a43685408", "subnet-0930151ba3b01750a", "subnet-0a7a978cd1d8837d3"]
  domains = [
    {
      name    = "eventbus.xarth.tv"
      zone_id = "Z09208311GBTCD9DO4I4F"
    }
  ]
}

module "controlplane_updater" {
  source = "../../modules/controlplane_updater_lambda"

  environment = local.environment
  account_id  = local.account_id
}

resource "aws_kms_key" "encryption_at_rest_key" {
  lifecycle {
    ignore_changes = [policy]
  }

  description = "SNS and SQS encryption at rest CMK"

  policy = <<EOF
{
    "Version": "2008-10-17",
    "Statement": [
        {
            "Sid": "Enable IAM User Permissions - EventBus",
            "Effect": "Allow",
            "Principal": {
                "AWS": "arn:aws:iam::${local.account_id}:root"
            },
            "Action": "kms:*",
            "Resource": "*"
        },
        {
            "Sid": "Enable IAM User Permissions - Publishers",
            "Effect": "Allow",
            "Principal": {
                "AWS": [
                    "arn:aws:iam::${local.account_id}:root"
                ]
            },
            "Action": [
                "kms:Decrypt",
                "kms:GenerateDataKey"
            ],
            "Resource": "*"
        },
        {
            "Sid": "Enable Service Permissions",
            "Effect": "Allow",
            "Principal": {
                "Service": [
                    "sqs.amazonaws.com",
                    "sns.amazonaws.com"
                ]
            },
            "Action": [
                "kms:Decrypt",
                "kms:GenerateDataKey"
            ],
            "Resource": "*"
        }
    ]
}
EOF

}

resource "aws_kms_alias" "encryption_at_rest_key" {
  name          = "alias/sns-sqs-encryption-key"
  target_key_id = aws_kms_key.encryption_at_rest_key.key_id
}

resource "aws_kms_key" "field_level_encryption_key" {
  description = "Key used to provide field-level encryption and authorization for EventBus events"

  policy = <<EOF
{
    "Version": "2008-10-17",
    "Statement": [
        {
            "Sid": "Enable IAM User Permissions - EventBus",
            "Effect": "Allow",
            "Principal": {
                "AWS": "arn:aws:iam::${local.account_id}:root"
            },
            "Action": "kms:*",
            "Resource": "*"
        }
    ]
}
EOF

}

resource "aws_kms_alias" "field_level_encryption_key_alias" {
  name          = "alias/field-level-encryption-key"
  target_key_id = aws_kms_key.field_level_encryption_key.key_id
}

module "alarms" {
  source                          = "../../modules/alarms"
  environment                     = local.environment
  high_urgency_key                = "a0909dc609874c6ebaa622236819a0dc"
  low_urgency_key                 = "898b319778b444c1b154f47f639760a7"
  controlplane_httpserver_alb_arn = module.app.controlplane_httpserver_alb_arn
  dashboard_alb_arn               = module.app.dashboard_alb_arn
  kms_crypto_api_max_rps          = 30000
}

module "cross_account_setup" {
  source = "../../modules/cross_account_setup"

  environment                   = local.environment
  account_id                    = local.account_id
  s3_bucket                     = local.s3_bucket_name
  encryption_at_rest_cmk_key_id = aws_kms_key.encryption_at_rest_key.key_id
  authorized_fields_cmk_key_id  = aws_kms_key.field_level_encryption_key.key_id
  latest_version                = "v1.4.0"
}

module "rds" {
  source      = "../../modules/rds"
  environment = local.environment
}

module "db_cross_region_backup" {
  source = "../../modules/db_backup"

  environment = local.environment
  db_arn      = module.rds.cluster_arn 
}
