provider "aws" {
  version = "~> 2.0"
  region  = "us-west-2"
}

locals {
  region                              = "us-west-2"
  environment                         = "staging"
  account_name                        = "twitch-eventbus-dev"
  account_id                          = "297385687169"
  s3_bucket_name                      = "eventbus-config-testing.internal.justin.tv"
  controlplane_httpserver_task_memory = 1024
  controlplane_httpserver_task_cpu    = 512
  controlplane_converger_task_memory  = 1024
  controlplane_converger_task_cpu     = 512
  instance_type                       = "m5.large"
  controlplane_httpserver_app_count   = 2
  controlplane_converger_app_count    = 1
  dashboard_task_memory               = 1024
  dashboard_task_cpu                  = 512
  dashboard_app_count                 = 1

  controlplane_validator_task_memory   = 1024
  controlplane_validator_task_cpu      = 512
  controlplane_validator_app_count     = 1
  controlplane_validator_slack_channel = "#eventbus-validation-staging"
}

terraform {
  required_version = ">= 0.11"

  backend "s3" {
    bucket = "twitch-eventbus-dev"
    key    = "tfstate/eventbus/twitch-eventbus-dev/terraform.state"
    region = "us-west-2"
  }
}

data "terraform_remote_state" "account" {
  backend = "s3"

  config = {
    bucket = "twitch-eventbus-dev"
    key    = "tfstate/eventbus/twitch-eventbus-dev/account/terraform.tfstate"
    region = "us-west-2"
  }
}

module "app" {
  source = "../../modules/app"

  environment                         = local.environment
  region                              = local.region
  controlplane_httpserver_image       = "${local.account_id}.dkr.ecr.${local.region}.amazonaws.com/controlplane-httpserver:latest"
  controlplane_converger_image        = "${local.account_id}.dkr.ecr.${local.region}.amazonaws.com/controlplane-converger:latest"
  dashboard_image                     = "${local.account_id}.dkr.ecr.${local.region}.amazonaws.com/dashboard:latest"
  ec2_instance_type                   = local.instance_type
  cluster_size                        = var.cluster_size
  controlplane_httpserver_app_count   = local.controlplane_httpserver_app_count
  controlplane_httpserver_task_memory = local.controlplane_httpserver_task_memory
  controlplane_httpserver_task_cpu    = local.controlplane_httpserver_task_cpu
  controlplane_converger_app_count    = local.controlplane_converger_app_count
  controlplane_converger_task_memory  = local.controlplane_converger_task_memory
  controlplane_converger_task_cpu     = local.controlplane_converger_task_cpu
  dashboard_app_count                 = local.dashboard_app_count
  dashboard_task_memory               = local.dashboard_task_memory
  dashboard_task_cpu                  = local.dashboard_task_cpu
  rds_reader_hostname                 = module.rds.reader_hostname
  rds_writer_hostname                 = module.rds.writer_hostname
  authorized_field_key_arn            = aws_kms_key.field_level_encryption_key.arn
  encryption_at_rest_key_arn          = aws_kms_key.encryption_at_rest_key.arn

  controlplane_validator_image         = "${local.account_id}.dkr.ecr.${local.region}.amazonaws.com/controlplane-validator:latest"
  controlplane_validator_app_count     = local.controlplane_validator_app_count
  controlplane_validator_task_memory   = local.controlplane_validator_task_memory
  controlplane_validator_task_cpu      = local.controlplane_validator_task_cpu
  controlplane_validator_slack_channel = local.controlplane_validator_slack_channel
}

module "controlplane_updater" {
  source = "../../modules/controlplane_updater_lambda"

  environment = local.environment
  account_id  = local.account_id
}

resource "aws_kms_key" "encryption_at_rest_key" {
  lifecycle {
    ignore_changes = [policy]
  }

  description = "SNS and SQS encryption at rest CMK"

  policy = <<EOF
{
    "Version": "2008-10-17",
    "Statement": [
        {
            "Sid": "Enable IAM User Permissions - EventBus",
            "Effect": "Allow",
            "Principal": {
                "AWS": "arn:aws:iam::${local.account_id}:root"
            },
            "Action": "kms:*",
            "Resource": "*"
        },
        {
            "Sid": "Enable IAM User Permissions - Publishers",
            "Effect": "Allow",
            "Principal": {
                "AWS": [
                    "arn:aws:iam::${local.account_id}:root"
                ]
            },
            "Action": [
                "kms:Decrypt",
                "kms:GenerateDataKey"
            ],
            "Resource": "*"
        },
        {
            "Sid": "Enable Service Permissions",
            "Effect": "Allow",
            "Principal": {
                "Service": [
                    "sqs.amazonaws.com",
                    "sns.amazonaws.com"
                ]
            },
            "Action": [
                "kms:Decrypt",
                "kms:GenerateDataKey"
            ],
            "Resource": "*"
        }
    ]
}
EOF

}

resource "aws_kms_alias" "encryption_at_rest_key" {
  name          = "alias/sns-sqs-encryption-key"
  target_key_id = aws_kms_key.encryption_at_rest_key.key_id
}

resource "aws_kms_key" "field_level_encryption_key" {
  description = "Key used to provide field-level encryption and authorization for EventBus events"

  policy = <<EOF
{
    "Version": "2008-10-17",
    "Statement": [
        {
            "Sid": "Enable IAM User Permissions - EventBus",
            "Effect": "Allow",
            "Principal": {
                "AWS": "arn:aws:iam::${local.account_id}:root"
            },
            "Action": "kms:*",
            "Resource": "*"
        }
    ]
}
EOF

}

resource "aws_kms_alias" "field_level_encryption_key_alias" {
  name          = "alias/field-level-encryption-key"
  target_key_id = aws_kms_key.field_level_encryption_key.key_id
}

module "cross_account_setup" {
  source = "../../modules/cross_account_setup"

  environment                   = local.environment
  account_id                    = local.account_id
  s3_bucket                     = local.s3_bucket_name
  encryption_at_rest_cmk_key_id = aws_kms_key.encryption_at_rest_key.key_id
  authorized_fields_cmk_key_id  = aws_kms_key.field_level_encryption_key.key_id
  latest_version                = "v1.4.0"
}

module "osiris_stack_eventbus_dashboard" {
  source                = "../../modules/osiris_stack"
  osiris_service_name   = "eventbus-dashboard-staging"
  team_name             = "Edge Infrastructure"
  team_email            = "eventbus@twitch.tv"
  additional_domains    = ["eventbus-staging.internal.justin.tv", "eventbus-staging.xarth.tv"]
  internal_fqdn         = "dashboard.staging.eventbus.twitch.a2z.com"
  osiris_pf_integration = "EventbusDashboardStaging-osiris_IDP"
  sandstorm_role        = "osiris-eventbus-dashboard-staging"
  sandstorm_prefix      = "eventbus/dashboard/staging"
  vpc_id                = "vpc-0134f9e6f9e3d2189"
  public_subnets        = ["subnet-0af6da1c688648098", "subnet-080d2c3a463bf0ffe", "subnet-0fbeb2c55196c1b99"]
  private_subnets       = ["subnet-00794dca198f8a455", "subnet-04252e1fe6be67db9", "subnet-0f30568646320f498"]
  domains = [
    {
      zone_id = "Z09194231TPOW6O9BFPBC"
      name    = "eventbus-staging.xarth.tv"
    }
  ]
}

module "rds" {
  source      = "../../modules/rds"
  environment = local.environment
}

module "db_cross_region_backup" {
  source = "../../modules/db_backup"

  environment = local.environment
  db_arn      = module.rds.cluster_arn 
}