locals {
  # Minimum 100 events in 5 minutes
  high_frequency_events = {
    authorization_create       = "AuthorizationCreate"
    channel_update             = "ChannelUpdate"
    content_tags_update        = "ContentTagsUpdate"
    entitlement_emote_create   = "EntitlementEmoteCreate"
    stream_create              = "StreamCreate"
    stream_delete              = "StreamDelete"
    stream_update              = "StreamUpdate"
    user_block_user_create     = "UserBlockUserCreate"
    user_follow_user_create    = "UserFollowUserCreate"
    user_subscribe_user_update = "UserSubscribeUserUpdate"
    user_update                = "UserUpdate"
  }

  # Minimum 1 event in 5 minutes
  low_frequency_events = {
    active_subscription_create                     = "ActiveSubscriptionCreate"
    active_subscription_delete                     = "ActiveSubscriptionDelete"
    authorization_delete                           = "AuthorizationDelete"
    automod_chat_message_caught_create             = "AutomodChatMessageCaughtCreate"
    automod_terms_moderation_action_create         = "AutomodTermsModerationActionCreate"
    channel_brand_safety_update                    = "ChannelBrandSafetyUpdate"
    channel_points_custom_reward_redemption_create = "ChannelPointsCustomRewardRedemptionCreate"
    channel_points_custom_reward_redemption_update = "ChannelPointsCustomRewardRedemptionUpdate"
    channel_points_custom_reward_update            = "ChannelPointsCustomRewardUpdate"
    chat_ban_create                                = "ChatBanCreate"
    chat_moderation_action_create                  = "ChatModerationActionCreate"
    channel_moderator_create                       = "ChannelModeratorCreate"
    chat_unban_request_create                      = "ChatUnbanRequestCreate"
    chat_unban_request_update                      = "ChatUnbanRequestUpdate"
    cheer_create                                   = "CheerCreate"
    clock_update                                   = "ClockUpdate"
    commercial_update                              = "CommercialUpdate"
    entitlement_drop_create                        = "EntitlementDropCreate"
    entitlement_emote_delete                       = "EntitlementEmoteDelete"
    entitlement_emote_update                       = "EntitlementEmoteUpdate"
    host_moderation_action_create                  = "HostModerationActionCreate"
    poll_update                                    = "PollUpdate"
    prediction_event_update                        = "PredictionEventUpdate"
    promo_campaign_update                          = "PromoCampaignUpdate"
    promotion_impression_create                    = "PromotionImpressionCreate"
    user_create                                    = "UserCreate"
    user_gift_subscription_user_create             = "UserGiftSubscriptionUserCreate"
    user_password_update                           = "UserPasswordUpdate"
    user_subscribe_user_notice_create              = "UserSubscribeUserNoticeCreate"
  }

  # No events in 5 minutes. They are included so that we can verify that *all* events have either been alarmed or not-alarmed intentionally.
  very_low_frequency_events = {
    automod_chat_message_caught_update        = "AutomodChatMessageCaughtUpdate"
    automod_settings_moderation_action_create = "AutomodSettingsModerationActionCreate"
    ban_evasion_detection_settings_update     = "BanEvasionDetectionSettingsUpdate"
    bits_use_create                           = "BitsUseCreate"
    boost_order_create                        = "BoostOrderCreate"
    boost_order_delete                        = "BoostOrderDelete"
    boost_order_update                        = "BoostOrderUpdate"
    bounty_status_update                      = "BountyStatusUpdate"
    category_create                           = "CategoryCreate"
    category_delete                           = "CategoryDelete"
    category_update                           = "CategoryUpdate"
    channel_ad_free_benefit_update            = "ChannelAdFreeBenefitUpdate"
    channel_endorsement_list_update           = "ChannelEndorsementListUpdate"
    channel_moderator_create                  = "ChannelModeratorCreate"
    channel_moderator_delete                  = "ChannelModeratorDelete"
    channel_points_custom_reward_create       = "ChannelPointsCustomRewardCreate"
    channel_points_custom_reward_delete       = "ChannelPointsCustomRewardDelete"
    chat_ban_delete                           = "ChatBanDelete"
    creator_subscription_settings_update      = "CreatorSubscriptionSettingsUpdate"
    emoticon_create                           = "EmoticonCreate"
    emoticon_delete                           = "EmoticonDelete"
    entitlement_badge_create                  = "EntitlementBadgeCreate"
    entitlement_badge_delete                  = "EntitlementBadgeDelete"
    entitlement_badge_update                  = "EntitlementBadgeUpdate"
    entitlement_drop_batch_create             = "EntitlementDropBatchCreate"
    entitlement_subscription_create           = "EntitlementSubscriptionCreate"
    entitlement_subscription_delete           = "EntitlementSubscriptionDelete"
    entitlement_subscription_update           = "EntitlementSubscriptionUpdate"
    host_moderation_action_delete             = "HostModerationActionDelete"
    hype_train_create                         = "HypeTrainCreate"
    hype_train_delete                         = "HypeTrainDelete"
    hype_train_update                         = "HypeTrainUpdate"
    idgb_game_update                          = "IgdbGameUpdate"
    igdb_game_create                          = "IgdbGameCreate"
    igdb_game_delete                          = "IgdbGameDelete"
    poll_create                               = "PollCreate"
    user_poll_sync_update                     = "PollSynchronizationUpdate"
    prediction_event_create                   = "PredictionEventCreate"
    raid_moderation_action_create             = "RaidModerationActionCreate"
    raid_moderation_action_delete             = "RaidModerationActionDelete"
    raid_update                               = "RaidUpdate"
    shared_ban_information_settings_update    = "SharedBanInformationSettingsUpdate"
    subscriber_badge_create                   = "SubscriberBadgeCreate"
    subscriber_badge_flair_create             = "SubscriberBadgeFlairCreate"
    tag_create                                = "TagCreate"
    tag_update                                = "TagUpdate"
    user_ban_create                           = "UserBanCreate"
    user_ban_delete                           = "UserBanDelete"
    user_data_request_create                  = "UserDataRequestCreate"
    user_delete                               = "UserDelete"
    user_destroy                              = "UserDestroy"
    user_multi_factor_auth_update             = "UserMultiFactorAuthUpdate"
    user_session_delete                       = "UserSessionDelete" # note: migrate back to low_frequency_events when new Session svc is publishing again
    social_media_link_create                  = "UserSocialMediaLinkCreate"
    social_media_link_delete                  = "UserSocialMediaLinkDelete"
    social_media_link_update                  = "UserSocialMediaLinkUpdate"
    user_undelete                             = "UserUndelete"

    # Events for which the schema has been added but have not yet begun publishing - these should be added to above categories once publishing
    ads_creator_incentive_offer_set_create     = "AdsCreatorIncentiveOfferSetCreate"
    channel_points_community_goal_update       = "ChannelPointsCommunityGoalUpdate"
    chat_verification_settings_update          = "ChatVerificationSettingsUpdate"
    creator_goal_create                        = "CreatorGoalCreate"
    creator_goal_update                        = "CreatorGoalUpdate"
    low_trust_user_treatment_update            = "LowTrustUserTreatmentUpdate"
    multiplayer_ads_bits_reward_create         = "MultiplayerAdsBitsRewardCreate"
    shared_ban_information_relationship_create = "SharedBanInformationRelationshipCreate"
    shared_ban_information_relationship_delete = "SharedBanInformationRelationshipDelete"
    user_recover_create                        = "UserRecoverCreate"
    
  }
}

resource "aws_cloudwatch_metric_alarm" "high_frequency_alarm" {
  for_each = local.high_frequency_events

  alarm_name                = "high-frequency-alarm-${each.key}"
  comparison_operator       = "LessThanOrEqualToThreshold"
  evaluation_periods        = "1"
  threshold                 = "100"
  alarm_description         = "event stream has published less than expected number of events"
  insufficient_data_actions = []
  ok_actions                = ["${module.pagerduty.pd_low_urgency_arn}"]
  alarm_actions             = ["${module.pagerduty.pd_low_urgency_arn}"]

  metric_query {
    id          = each.key
    return_data = true

    metric {
      metric_name = "NumberOfMessagesPublished"
      namespace   = "AWS/SNS"
      period      = "300"
      stat        = "Sum"
      unit        = "Count"

      dimensions = {
        TopicName = "eventbus-production-${each.value}"
      }
    }
  }
}

resource "aws_cloudwatch_metric_alarm" "low_frequency_alarm" {
  for_each = local.low_frequency_events

  alarm_name                = "low-frequency-alarm-${each.key}"
  comparison_operator       = "LessThanOrEqualToThreshold"
  evaluation_periods        = "1"
  threshold                 = "1"
  alarm_description         = "event stream has published less than expected number of events"
  insufficient_data_actions = []
  ok_actions                = ["${module.pagerduty.pd_low_urgency_arn}"]
  alarm_actions             = ["${module.pagerduty.pd_low_urgency_arn}"]

  metric_query {
    id          = each.key
    return_data = true

    metric {
      metric_name = "NumberOfMessagesPublished"
      namespace   = "AWS/SNS"
      period      = "300"
      stat        = "Sum"
      unit        = "Count"

      dimensions = {
        TopicName = "eventbus-production-${each.value}"
      }
    }
  }
}

resource "aws_cloudwatch_metric_alarm" "all_events_alarm" {
  alarm_name                = "all-events-alarm"
  comparison_operator       = "LessThanOrEqualToThreshold"
  evaluation_periods        = "1"
  threshold                 = "1"
  alarm_description         = "all high frequency event streams on event bus have published zero events"
  insufficient_data_actions = []
  ok_actions                = ["${module.pagerduty.pd_high_urgency_arn}"]
  alarm_actions             = ["${module.pagerduty.pd_high_urgency_arn}"]

  metric_query {
    id          = "all_events_alarm"
    return_data = true

    // we cannot use more than 10 metrics in a composite query
    expression  = join(" + ", toset(slice(keys(local.high_frequency_events), 0, 10)))
  }

  dynamic "metric_query" {
    // we cannot use more than 10 metrics in a composite query
    for_each = toset(slice(keys(local.high_frequency_events), 0, 10))

    content {
      id = metric_query.key

      metric {
        metric_name = "NumberOfMessagesPublished"
        namespace   = "AWS/SNS"
        period      = "300"
        stat        = "Sum"
        unit        = "Count"

        dimensions = {
          TopicName = "eventbus-production-${metric_query.value}"
        }
      }
    }
  }
}
