data "aws_subnet_ids" "subnet_ids" {
  vpc_id = var.vpc_id
}

resource "aws_ecr_repository" "badges_ecr_repository" {
  name = "badges"
}

resource "aws_ecr_lifecycle_policy" "badges_lifecycle_policy" {
  repository = aws_ecr_repository.badges_ecr_repository.name

  policy = <<EOF
{
    "rules": [
        {
            "rulePriority": 1,
            "description": "Keep 30 most recent images",
            "selection": {
                "tagStatus": "any",
                "countType": "imageCountMoreThan",
                "countNumber": 30
            },
            "action": {
                "type": "expire"
            }
        }
    ]
}
EOF

}

resource "aws_s3_bucket" "badges_bucket" {
  bucket        = var.s3_bucket
  acl           = "private"
  force_destroy = false
}

resource "aws_iam_role" "badges_ecs_instance_role" {
  name = "badges-ecs-instance-role"

  assume_role_policy = <<EOF
{
    "Version": "2012-10-17",
    "Statement": [
        {
            "Action": "sts:AssumeRole",
            "Effect": "Allow",
            "Principal": {
                "Service": "ec2.amazonaws.com"
            }
        }
    ]
}
EOF

}

resource "aws_iam_role_policy_attachment" "badges_ecs_instance_role_policy_attachment" {
  role       = aws_iam_role.badges_ecs_instance_role.name
  policy_arn = "arn:aws:iam::aws:policy/service-role/AmazonEC2ContainerServiceforEC2Role"
}

resource "aws_iam_policy" "badges_policy" {
  name = "badges-policy"

  policy = <<EOF
{
    "Version": "2012-10-17",
    "Statement": [
        {
            "Effect": "Allow",
            "Action": "s3:*",
            "Resource": ["arn:aws:s3:::${var.s3_bucket}"]
        },
        {
            "Effect": "Allow",
            "Action": "s3:*",
            "Resource": ["arn:aws:s3:::${var.s3_bucket}/*"]
        },
        {
            "Effect": "Allow",
            "Action": [
              "cloudwatch:ListMetrics",
              "cloudwatch:GetMetricStatistics"
            ],
            "Resource": "*"
        }
    ]
}
EOF

}

resource "aws_iam_role_policy_attachment" "badges_runtime_role_policy_attachment" {
  role       = aws_iam_role.badges_ecs_instance_role.name
  policy_arn = aws_iam_policy.badges_policy.arn
}

resource "aws_iam_instance_profile" "badges_instance_profile" {
  name = "badges-instance-profile"
  role = aws_iam_role.badges_ecs_instance_role.name
}

resource "aws_iam_role" "badges_aws_batch_service_role" {
  name = "badges-runtime-role"

  assume_role_policy = <<EOF
{
    "Version": "2012-10-17",
    "Statement": [
        {
            "Action": "sts:AssumeRole",
            "Effect": "Allow",
            "Principal": {
                "Service": "batch.amazonaws.com"
            }
        }
    ]
}
EOF

}

resource "aws_iam_role_policy_attachment" "badges_aws_batch_service_role" {
  role       = aws_iam_role.badges_aws_batch_service_role.name
  policy_arn = "arn:aws:iam::aws:policy/service-role/AWSBatchServiceRole"
}

resource "aws_batch_compute_environment" "badges_compute_environment" {
  compute_environment_name = "badges"

  compute_resources {
    instance_role = aws_iam_instance_profile.badges_instance_profile.arn

    instance_type = [
      "c5.large",
    ]

    max_vcpus = 4
    min_vcpus = 0

    security_group_ids = var.security_group_ids

    subnets = data.aws_subnet_ids.subnet_ids.ids

    type = "EC2"
  }

  service_role = aws_iam_role.badges_aws_batch_service_role.arn
  type         = "MANAGED"
  depends_on   = [aws_iam_role_policy_attachment.badges_aws_batch_service_role]
}

resource "aws_batch_job_queue" "badges_job_queue" {
  name                 = "badges-job-queue"
  state                = "ENABLED"
  priority             = 1
  compute_environments = [aws_batch_compute_environment.badges_compute_environment.arn]
}

resource "aws_batch_job_definition" "badges_job_definition" {
  name = "badges-job"
  type = "container"

  container_properties = <<CONTAINER_PROPERTIES
{
    "command": ["/generate.sh", "${var.environment}"],
    "image": "${aws_ecr_repository.badges_ecr_repository.repository_url}:latest",
    "memory": 1024,
    "vcpus": 1,
    "volumes": [],
    "environment": [
        {"name": "ENVIRONMENT", "value": "${var.environment}"},
        {"name": "S3_BUCKET", "value": "${var.s3_bucket}"}
    ],
    "mountPoints": [],
    "ulimits": []
}
CONTAINER_PROPERTIES

}

// Run the batch job daily
resource "aws_iam_role" "badges_cron_role" {
  name = "badges-cron-role"

  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Effect": "Allow",
      "Principal": {
        "Service": "events.amazonaws.com"
      },
      "Action": "sts:AssumeRole"
    }
  ]
}
EOF

}

resource "aws_iam_policy" "badges_cron_policy" {
  name = "badges-cron-policy"

  policy = <<EOF
{
    "Version": "2012-10-17",
    "Statement": [
        {
            "Effect": "Allow",
            "Action": "batch:SubmitJob",
            "Resource": "*"
        }
    ]
}
EOF

}

resource "aws_iam_role_policy_attachment" "badges_cron_policy_attachment" {
  role       = aws_iam_role.badges_cron_role.name
  policy_arn = aws_iam_policy.badges_cron_policy.arn
}

resource "aws_cloudwatch_event_rule" "badges_cron_rule" {
  name        = "BadgesCron"
  description = "Run the badge generation script daily"

  // Run every day at 16:00 UTC (8 or 9AM PST)
  // Docs: https://docs.aws.amazon.com/AmazonCloudWatch/latest/events/ScheduledEvents.html
  schedule_expression = "cron(0 16 * * ? *)"
}

resource "aws_cloudwatch_event_target" "badges_cron_target" {
  rule     = aws_cloudwatch_event_rule.badges_cron_rule.name
  arn      = aws_batch_job_queue.badges_job_queue.arn
  role_arn = aws_iam_role.badges_cron_role.arn

  batch_target {
    job_definition = aws_batch_job_definition.badges_job_definition.arn
    job_name       = "badges-daily"
  }
}

